#!/bin/sh
#
# Generates a full project for a HeadersTest app.
# This is a compile-time test checking if all the installed headers can be included individually.
#
# Example for KDb:
# % ./headers_test.sh ~/kde/build/kdb ~/kde/src/kdb KDb kdb .
#
# It is assumed that this script is executed from the current build subdirectory.
# Source dir goes to the app/ subdirectory, build dir goes to the app-build subdirectory.
# Nothing is cached, both dirs are fully recreated on each run.

set -e
builddir=$1
current_builddir=`pwd`
shift
srcdir=$1

if [ ! -d "$builddir" -o ! -d "$srcdir" -o $# -lt 4 ] ; then
    echo "Usage: $me <project_binary_dir> <project_source_dir> <libraries_to_link> <project_name_prefix> <subdirs_to_find>..."
    exit 1
fi

test_app_dir=$current_builddir/app
rm -rf $test_app_dir
mkdir -p $test_app_dir

test_app_builddir=$current_builddir/app-build
rm -rf $test_app_builddir
mkdir -p $test_app_builddir

me=headers_test.sh
current_srcdir=`dirname $0`

shift
link_libs=$1
shift
prefix=$1
shift

cd $srcdir

cat <<EOT > $test_app_dir/CMakeLists.txt
# Generated by $current_srcdir/$me
# prefix: $prefix
# link_libs: $link_libs
# subdirs: $@
#
# WARNING! All changes made in this file will be lost!
#
# Test if all the installed headers can be included individually.
# This is a compile-time test.

cmake_minimum_required(VERSION 2.8.12 FATAL_ERROR)
find_package(ECM 1.8.0 REQUIRED NO_MODULE)
set(CMAKE_MODULE_PATH $current_srcdir $srcdir/cmake/modules \${ECM_MODULE_PATH} \${ECM_KDE_MODULE_DIR})

project(HeadersTest)
include(HeadersTestInclude)

add_executable(
    HeadersTest
    HeadersTest.cpp
EOT

cat <<EOT > $test_app_dir/HeadersTest.cpp
// Generated by $me
// WARNING! All changes made in this file will be lost!

// Nothing to do here
int main(int, char**)
{
    return 0;
}
EOT

# files to include using <name>, forward headers + some *.h headers
find_files()
{
    for subdir in "$@" ; do
        find "$builddir/src/$subdir" -maxdepth 1 -type f -printf "%f\n"
    done
}

for f in `find_files $@ | grep -v "\.h\$" | grep -vE "(\\.|Makefile)" | sort`; do
    fname=${f}_HeaderTest.cpp
    echo "#include <$f>" > $test_app_dir/$fname
    echo "    $fname" >> $test_app_dir/CMakeLists.txt
done

# files to include using <name>, these are .h files
for f in `find_files $@ | grep "\.h\$" | grep -vE "(^ui_.*\.h|sqlparser\.h)" | sort`; do
    fname=${f}_HeaderTest.cpp
    echo "#include <$f>" > $test_app_dir/$fname
    echo "    $fname" >> $test_app_dir/CMakeLists.txt
done

cat <<EOT >> $test_app_dir/CMakeLists.txt
)

target_link_libraries(HeadersTest
    PUBLIC
        $link_libs
)

feature_summary(WHAT ALL FATAL_ON_MISSING_REQUIRED_PACKAGES)
EOT

# Finally, build:
cd $test_app_builddir
cmake ../app
make
