// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F460

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_80_3_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x9d787891u, 0xfc63d657u, 0, 46 }, // 9d787891fc63d657 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__16_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x13b6d865u, 0x0ff65a64u, 0, 63 }, // 13b6d8650ff65a64 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x58be57a8u, 0x3f1203a2u, 0, 80 }, // 58be57a83f1203a2 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x18ba8118u, 0x8a1a7c6eu, 0, 6 }, // 18ba81188a1a7c6e = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xeee5ee2fu, 0x726485e1u, 0, 155 }, // eee5ee2f726485e1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xfc401771u, 0x402ba34bu, 40, 46 }, // fc401771402ba34b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xbf785603u, 0xec4dc1b2u, 40, 138 }, // bf785603ec4dc1b2 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__32_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x719da7bbu, 0x8ea4d7f3u, 40, 63 }, // 719da7bb8ea4d7f3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x05f59702u, 0x58ef6d32u, 40, 80 }, // 05f5970258ef6d32 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x93c31e56u, 0xec771634u, 40, 6 }, // 93c31e56ec771634 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x5c31e48du, 0x69f7a8d2u, 97, 46 }, // 5c31e48d69f7a8d2 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp32@16_80_3_T_F_0___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{ 4, 3, 4, 4, 4, 4, 1, 0, 4, 2},
 { 1, 4, 4, 4, 4, 4, 4, 4, 4, 4},
 { 7, 7, 6, 6, 6, 6, 4, 6, 6, 8},
 { 0, 0, 0, 3, 0, 0, 0, 3, 0, 0},
 { 3, 7, 7, 7, 7, 7, 7, 7, 7, 7},
 { 1, 9, 9, 9, 9, 9, 9, 9, 9, 9},
 { 7, 5, 5, 9, 9, 9, 9, 9, 9, 9},
 { 5, 5, 7, 9, 9, 9, 9, 9, 9, 6},
 { 5, 5, 7, 9, 9, 9, 9, 9,10, 3},
 { 5, 7, 7, 9, 9, 9, 9, 7, 7, 7}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F460 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 80
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 80 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

