// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A7__F4

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_16_0_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1152)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xe5998ff4u, 0xff94b67au, 0, 46 }, // e5998ff4ff94b67a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1152
    { 0x1579afd5u, 0x60a4d817u, 40, 46 }, // 1579afd560a4d817 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1152
    { 0x15996810u, 0xf54652ebu, 40, 63 }, // 15996810f54652eb = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1152
    { 0xeeecfbd1u, 0xfaa59d72u, 40, 6 }, // eeecfbd1faa59d72 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1152
    { 0xbc5ea0beu, 0x6b1148ccu, 40, 23 }, // bc5ea0be6b1148cc = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1152
    { 0xfa838038u, 0xe862811eu, 97, 138 }, // fa838038e862811e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1152
    { 0xc7b5d451u, 0xabd59b77u, 97, 80 }, // c7b5d451abd59b77 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1152
    { 0xa59d3b62u, 0x2a50026cu, 97, 155 }, // a59d3b622a50026c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1152
    { 0x78100d4du, 0x84ee9b69u, 97, 103 }, // 78100d4d84ee9b69 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__32_32__CO__wave4_warp4_stg1--Arch_gfx1152
    { 0x878ddf10u, 0x5e313e51u, 120, 80 }, // 878ddf105e313e51 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1152
    { 0xa411b450u, 0x5505e6a7u, 126, 155 }, // a411b4505505e6a7 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_T_F_0___gfx11xx__P__64_32__CO__wave3_warp4_stg1--Arch_gfx1152
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1152_mod0
{{ 1, 7, 7, 5, 8, 7, 0, 0, 0, 0},
 { 9, 0, 7, 6, 8, 6, 6, 8, 8, 8},
 { 2, 2, 2, 2, 2, 4, 4, 2, 4,10},
 { 3, 1, 4, 4, 2, 1, 2, 1, 1, 1},
 { 1, 4, 3, 4, 1, 2, 3, 3, 1, 3},
 { 3, 1, 4, 4, 4, 3, 1, 4, 2, 2},
 { 3, 4, 3, 3, 3, 3, 3, 4, 2, 3},
 { 4, 4, 1, 4, 1, 4, 4, 3, 2, 4},
 { 4, 2, 2, 3, 4, 4, 1, 1, 2, 4},
 { 3, 3, 4, 4, 2, 1, 1, 3, 2, 1}}
// End of GPU gfx1152_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A7__F4 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

