#include "ngspice/cm.h"
extern void cm_cpmline(Mif_Private_t *);
/* ===========================================================================
	FILE    cfunc.mod for cm_cpmline
	Copyright 2025 Vadim Kuznetsov

	Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

	1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.

	2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.

	3. Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS IS” AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <complex.h>

#include "msline_common.h"
#include "tline_common.h"

#ifdef _MSC_VER
typedef _Dcomplex DoubleComplex;  // double complex
#else
typedef double complex DoubleComplex;
#endif

static double ae, ao, be, bo, ze, zo, ee, eo;

static void copy_complex(DoubleComplex s, Complex_t *d)
{
	d->real = creal(s);
	d->imag = cimag(s);
}

#ifdef _MSC_VER
static DoubleComplex divide(DoubleComplex n1, DoubleComplex n2)
    {
        DoubleComplex rez;
        double denom = n2._Val[0] * n2._Val[0] + n2._Val[1] * n2._Val[1];
        rez._Val[0] = (n1._Val[0] * n2._Val[0] + n1._Val[1] * n2._Val[1]) / denom;
        rez._Val[1] = (n1._Val[1] * n2._Val[0] - n1._Val[0] * n2._Val[1]) / denom;
        return rez;
    }

static DoubleComplex rdivide(double n1, DoubleComplex n2)
    {
        DoubleComplex rez;
        double denom = n2._Val[0] * n2._Val[0] + n2._Val[1] * n2._Val[1];
        rez._Val[0] = (n1 * n2._Val[0]) / denom;
        rez._Val[1] = (-1. * n1 * n2._Val[1]) / denom;
        return rez;
    }
#endif

//static cpline_state_t *state = NULL;
static void cm_cpmline_callback(Mif_Private_t *mif_private, Mif_Callback_Reason_t reason);

static void calcPropagation (double W, double s,
								 double er, double h, double t, double tand, double rho, double D,
								 int SModel, int DModel, double frequency)
{

	// quasi-static analysis
	double Zle, ErEffe, Zlo, ErEffo;
	cpmslineAnalyseQuasiStatic (W, h, s, t, er, SModel, &Zle, &Zlo, &ErEffe, &ErEffo);

	// analyse dispersion of Zl and Er
	double ZleFreq, ErEffeFreq, ZloFreq, ErEffoFreq;
	cpmslineAnalyseDispersion (W, h, s, t, er, Zle, Zlo, ErEffe, ErEffo, frequency, DModel,
					   &ZleFreq, &ZloFreq, &ErEffeFreq, &ErEffoFreq);

	// analyse losses of line
	double ace, aco, ade, ado;
	analyseLoss (W, t, er, rho, D, tand, Zle, Zlo, ErEffe,
						 frequency, HAMMERSTAD, &ace, &ade);
	analyseLoss (W, t, er, rho, D, tand, Zlo, Zle, ErEffo,
						 frequency, HAMMERSTAD, &aco, &ado);

	// compute propagation constants for even and odd mode
	double k0 = 2 * M_PI * frequency / C0;
	ae = ace + ade;
	ao = aco + ado;
	be = sqrt (ErEffeFreq) * k0;
	bo = sqrt (ErEffoFreq) * k0;
	ze = ZleFreq;
	zo = ZloFreq;
	ee = ErEffeFreq;
	eo = ErEffoFreq;
}



void cm_cpmline (Mif_Private_t *mif_private)
{
	Complex_t   z11, z12, z13, z14;

	/* how to get properties of this component, e.g. L, W */
	double W = mif_private->param[1]->element[0].rvalue;
	double l = mif_private->param[0]->element[0].rvalue;
	double s = mif_private->param[2]->element[0].rvalue;
	int SModel = mif_private->param[3]->element[0].ivalue;
	int DModel = mif_private->param[4]->element[0].ivalue;
    int TModel = mif_private->param[11]->element[0].ivalue;

	/* how to get properties of the substrate, e.g. Er, H */
	double er    = mif_private->param[5]->element[0].rvalue;
	double h     = mif_private->param[6]->element[0].rvalue;
	double t     = mif_private->param[7]->element[0].rvalue;
	double tand  = mif_private->param[8]->element[0].rvalue;
	double rho   = mif_private->param[9]->element[0].rvalue;
	double D     = mif_private->param[10]->element[0].rvalue;

    if(mif_private->circuit.init) {
        *(mif_private->callback) = cm_cpmline_callback;
        mif_private->inst_var[0]->element[0].pvalue = NULL;
	}


	/* Compute the output */
	if(mif_private->circuit.anal_type == DC) {
          calcPropagation(W,s,er,h,t,tand,rho,D,SModel,DModel,0);

		  double V1 = mif_private->conn[4]->port[0]->input.rvalue;
		  double V2 = mif_private->conn[5]->port[0]->input.rvalue;
		  double V3 = mif_private->conn[6]->port[0]->input.rvalue;
		  double V4 = mif_private->conn[7]->port[0]->input.rvalue;
		  double I1 = mif_private->conn[0]->port[0]->input.rvalue;
		  double I2 = mif_private->conn[1]->port[0]->input.rvalue;
		  double I3 = mif_private->conn[2]->port[0]->input.rvalue;
		  double I4 = mif_private->conn[3]->port[0]->input.rvalue;

		  double z = sqrt(ze*zo);

		  double V2out = V1 + z*I1;
		  double V1out = V2 + z*I2;
		  mif_private->conn[0]->port[0]->output.rvalue = V1out + I1*z;
		  mif_private->conn[1]->port[0]->output.rvalue = V2out + I2*z;

		  double V3out = V4 + z*I4;
		  double V4out = V3 + z*I3;
		  mif_private->conn[2]->port[0]->output.rvalue = V3out + I3*z;
		  mif_private->conn[3]->port[0]->output.rvalue = V4out + I4*z;

		  cm_analog_auto_partial();
	}
	else if(mif_private->circuit.anal_type == AC) {
		double o = mif_private->circuit.frequency;
        calcPropagation(W,s,er,h,t,tand,rho,D,SModel,DModel, o/(2*M_PI));
		DoubleComplex _Z11, _Z12, _Z13, _Z14;
#ifdef _MSC_VER
		DoubleComplex ge = _Cbuild(ae, be);
		DoubleComplex go = _Cbuild(ao, bo);
        DoubleComplex tango = _Cmulcr(ctanh(_Cmulcr(go, l)), 2.);
        DoubleComplex tange = _Cmulcr(ctanh(_Cmulcr(ge, l)), 2.);
        DoubleComplex singo = _Cmulcr(csinh(_Cmulcr(go, l)), 2.);
        DoubleComplex singe = _Cmulcr(csinh(_Cmulcr(ge, l)), 2.);
        DoubleComplex zotango = rdivide(zo, tango);
        DoubleComplex zetange = rdivide(ze, tange);
        DoubleComplex zosingo = rdivide(zo, singo);
        DoubleComplex zesinge = rdivide(ze, singe);

		_Z11._Val[0] = zotango._Val[0] + zetange._Val[0];
        _Z11._Val[1] = zotango._Val[1] + zetange._Val[1];
        _Z12._Val[0] = zosingo._Val[0] + zesinge._Val[0];
        _Z12._Val[1] = zosingo._Val[1] + zesinge._Val[1];
		_Z13._Val[0] = zesinge._Val[0] - zosingo._Val[0];
		_Z13._Val[1] = zesinge._Val[1] - zosingo._Val[1];
		_Z14._Val[0] = zetange._Val[0] - zotango._Val[0];
		_Z14._Val[1] = zetange._Val[1] - zotango._Val[1];
#else
		DoubleComplex ge =  ae + I*be;
		DoubleComplex go =  ao + I*bo;

		_Z11 = zo / (2*ctanh(go*l)) + ze / (2*ctanh(ge*l));
		_Z12 = zo / (2*csinh(go*l)) + ze / (2*csinh(ge*l));
		_Z13 = ze / (2*csinh(ge*l)) - zo / (2*csinh(go*l));
		_Z14 = ze / (2*ctanh(ge*l)) - zo / (2*ctanh(go*l));
#endif
		copy_complex(_Z11,&z11);
		copy_complex(_Z12,&z12);
		copy_complex(_Z13,&z13);
		copy_complex(_Z14,&z14);

        mif_private->conn[0]->port[0]->ac_gain[0].port[0] = z11; mif_private->conn[1]->port[0]->ac_gain[1].port[0] = z11;
        mif_private->conn[2]->port[0]->ac_gain[2].port[0] = z11; mif_private->conn[3]->port[0]->ac_gain[3].port[0] = z11;

		mif_private->conn[0]->port[0]->ac_gain[1].port[0] = z12; mif_private->conn[1]->port[0]->ac_gain[0].port[0] = z12;
        mif_private->conn[2]->port[0]->ac_gain[3].port[0] = z12; mif_private->conn[3]->port[0]->ac_gain[2].port[0] = z12;

		mif_private->conn[0]->port[0]->ac_gain[2].port[0] = z13; mif_private->conn[2]->port[0]->ac_gain[0].port[0] = z13;
        mif_private->conn[1]->port[0]->ac_gain[3].port[0] = z13; mif_private->conn[3]->port[0]->ac_gain[1].port[0] = z13;

		mif_private->conn[0]->port[0]->ac_gain[3].port[0] = z14; mif_private->conn[3]->port[0]->ac_gain[0].port[0] = z14;
        mif_private->conn[1]->port[0]->ac_gain[2].port[0] = z14; mif_private->conn[2]->port[0]->ac_gain[1].port[0] = z14;
	}
	else if(mif_private->circuit.anal_type == TRANSIENT) {
        calcPropagation(W,s,er,h,t,tand,rho,D,SModel,DModel,0);
		double time = mif_private->circuit.time;
		double Vp[PORT_NUM];
		double Ip[PORT_NUM];
		Vp[0] = mif_private->conn[4]->port[0]->input.rvalue;
		Vp[1] = mif_private->conn[5]->port[0]->input.rvalue;
		Vp[2] = mif_private->conn[6]->port[0]->input.rvalue;
		Vp[3] = mif_private->conn[7]->port[0]->input.rvalue;
		Ip[0] = mif_private->conn[0]->port[0]->input.rvalue;
		Ip[1] = mif_private->conn[1]->port[0]->input.rvalue;
		Ip[2] = mif_private->conn[2]->port[0]->input.rvalue;
		Ip[3] = mif_private->conn[3]->port[0]->input.rvalue;
		double delay = l/(C0);
        void **sim_points = &(mif_private->inst_var[0]->element[0].pvalue);
		if (TModel == TRAN_FULL) {
			cpline_state_t *last = get_cpline_last_state(*(cpline_state_t **)sim_points);
			double last_time = 0;
			if (last != NULL) last_time = last->time;

            if (mif_private->circuit.time < last_time) {
				delete_cpline_last_state((cpline_state_t **)sim_points);
			}
			append_cpline_state((cpline_state_t **)sim_points, time, Vp, Ip, 1.2*delay);
		}
        if (time > delay && TModel == TRAN_FULL) {
			cpline_state_t *pp = find_cpline_state(*(cpline_state_t **)sim_points, time - delay);
			if (pp != NULL) {

				double J1e = 0.5*(Ip[3] + Ip[0]);
				double J1o = 0.5*(Ip[0] - Ip[3]);
				double J2e = 0.5*(Ip[1] + Ip[2]);
				double J2o = 0.5*(Ip[1] - Ip[2]);


				double J1et = 0.5*(pp->Ip[3] + pp->Ip[0]);
				double J1ot = 0.5*(pp->Ip[0] - pp->Ip[3]);
				double J2et = 0.5*(pp->Ip[1] + pp->Ip[2]);
				double J2ot = 0.5*(pp->Ip[1] - pp->Ip[2]);


				double V1et = 0.5*(pp->Vp[3] + pp->Vp[0]);
				double V1ot = 0.5*(pp->Vp[0] - pp->Vp[3]);
				double V2et = 0.5*(pp->Vp[1] + pp->Vp[2]);
				double V2ot = 0.5*(pp->Vp[1] - pp->Vp[2]);

				double V1e = ze*J1e + V2et + ze*J2et;
				double V1o = zo*J1o + V2ot + zo*J2ot;
				double V2e = ze*J2e + V1et + ze*J1et;
				double V2o = zo*J2o + V1ot + zo*J1ot;

				double V1 = V1o + V1e;
				double V2 = V2o + V2e;
				double V3 = V2e - V2o;
				double V4 = V1e - V1o;

				mif_private->conn[0]->port[0]->output.rvalue = V1;
				mif_private->conn[1]->port[0]->output.rvalue = V2;
				mif_private->conn[2]->port[0]->output.rvalue = V3;
				mif_private->conn[3]->port[0]->output.rvalue = V4;
			}
			cm_analog_auto_partial();
		} else {
		    double z = sqrt(ze*zo);
			double V2out = Vp[0] + z*Ip[0];
			double V1out = Vp[1] + z*Ip[1];
			mif_private->conn[0]->port[0]->output.rvalue = V1out + Ip[0]*z;
			mif_private->conn[1]->port[0]->output.rvalue = V2out + Ip[1]*z;

			double V3out = Vp[3] + z*Ip[3];
			double V4out = Vp[2] + z*Ip[2];
			mif_private->conn[2]->port[0]->output.rvalue = V3out + Ip[2]*z;
			mif_private->conn[3]->port[0]->output.rvalue = V4out + Ip[3]*z;

			cm_analog_auto_partial();
		}
	}
}

static void cm_cpmline_callback(Mif_Private_t *mif_private, Mif_Callback_Reason_t reason)
{
    switch (reason) {
        case MIF_CB_DESTROY:
            delete_cpline_states((cpline_state_t **)&(mif_private->inst_var[0]->element[0].pvalue));
            break;
        default: break;
    }
}
