/*
 *      demod_ufsk12.c -- 1200 baud FSK demodulator
 *
 *      Copyright (C) 2007, 2024
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* ---------------------------------------------------------------------- */

#include "multimon.h"
#include "filter.h"
#include <math.h>
#include <string.h>

/* ---------------------------------------------------------------------- */

/*
 * Standard TCM3105 clock frequency: 4.4336MHz
 * Mark frequency: 2200 Hz
 * Space frequency: 1200 Hz
 */

#define FREQ_MARK  1200
#define FREQ_SPACE 2200
#define FREQ_SAMP  22050
#define BAUD       1200
#define SUBSAMP    2

/* ---------------------------------------------------------------------- */

#define CORRLEN ((int)(FREQ_SAMP/BAUD))
#define SPHASEINC (0x10000u*BAUD*SUBSAMP/FREQ_SAMP)

static float corr_mark_i[CORRLEN];
static float corr_mark_q[CORRLEN];
static float corr_space_i[CORRLEN];
static float corr_space_q[CORRLEN];

/* ---------------------------------------------------------------------- */
	
static void ufsk12_init(struct demod_state *s)
{
	float f;
	int i;

	uart_init(s);
	memset(&s->l1.ufsk12, 0, sizeof(s->l1.ufsk12));
	for (f = 0, i = 0; i < CORRLEN; i++) {
		corr_mark_i[i] = cos(f);
		corr_mark_q[i] = sin(f);
		f += 2.0*M_PI*FREQ_MARK/FREQ_SAMP;
	}
	for (f = 0, i = 0; i < CORRLEN; i++) {
		corr_space_i[i] = cos(f);
		corr_space_q[i] = sin(f);
		f += 2.0*M_PI*FREQ_SPACE/FREQ_SAMP;
	}
}

/* ---------------------------------------------------------------------- */

static void ufsk12_demod(struct demod_state *s, buffer_t buffer, int length)
{
	float f;
	unsigned char curbit;

	if (s->l1.ufsk12.subsamp) {
		if (length <= (int)s->l1.ufsk12.subsamp) {
			s->l1.ufsk12.subsamp -= length;
			return;
		}
		buffer.fbuffer += s->l1.ufsk12.subsamp;
		length -= s->l1.ufsk12.subsamp;
		s->l1.ufsk12.subsamp = 0;
	}
	for (; length > 0; length -= SUBSAMP, buffer.fbuffer += SUBSAMP) {
		f = 	fsqr(mac(buffer.fbuffer, corr_mark_i, CORRLEN)) +
			fsqr(mac(buffer.fbuffer, corr_mark_q, CORRLEN)) -
			fsqr(mac(buffer.fbuffer, corr_space_i, CORRLEN)) -
			fsqr(mac(buffer.fbuffer, corr_space_q, CORRLEN));
		s->l1.ufsk12.dcd_shreg <<= 1;
		s->l1.ufsk12.dcd_shreg |= (f > 0);
		verbprintf(10, "%c", '0'+(s->l1.ufsk12.dcd_shreg & 1));
		/*
		 * check if transition
		 */
		if ((s->l1.ufsk12.dcd_shreg ^ (s->l1.ufsk12.dcd_shreg >> 1)) & 1) {
			if (s->l1.ufsk12.sphase < (0x8000u-(SPHASEINC/2)))
				s->l1.ufsk12.sphase += SPHASEINC/8;
			else
				s->l1.ufsk12.sphase -= SPHASEINC/8;
		}
		s->l1.ufsk12.sphase += SPHASEINC;
		if (s->l1.ufsk12.sphase >= 0x10000u) {
			s->l1.ufsk12.sphase &= 0xffffu;
			curbit = s->l1.ufsk12.dcd_shreg & 1;
			verbprintf(9, " %c ", '0'+curbit);
			uart_rxbit(s, curbit);
		}
	}
	s->l1.ufsk12.subsamp = -length;
}

/* ---------------------------------------------------------------------- */

const struct demod_param demod_ufsk1200 = {
    "UFSK1200", true, FREQ_SAMP, CORRLEN, ufsk12_init, ufsk12_demod, NULL
};

/* ---------------------------------------------------------------------- */
