// SPDX-License-Identifier: GPL-2.0-or-later
// SPDX-FileCopyrightText: 2011 Konrad Twardowski

#pragma once

#include "config.h"

#include <QDialog>
#include <QDialogButtonBox>
#include <QVBoxLayout>

class InfoWidget;

class UDialog: public QDialog {
public:
	enum class WindowSize { FIXED, LARGE };

	explicit UDialog(QWidget *parent, const QString &windowTitle, const bool simple);
	virtual ~UDialog() = default;
	InfoWidget *infoWidget() { return m_infoWidget; }
	QPushButton *acceptButton() { return m_acceptButton; }
	QWidget *bottomWidget() { return m_bottomWidget; }
	QVBoxLayout *mainLayout() { return m_mainLayout; }

	void moveToCenterOfScreen();
	void setWindowSize(const WindowSize windowSize);

	// messages

	static void error(QWidget *parent, const QString &text, const Qt::TextFormat format = Qt::AutoText);
	static void info(QWidget *parent, const QString &text, const Qt::TextFormat format = Qt::AutoText);
	static void warning(QWidget *parent, const QString &text, const Qt::TextFormat format = Qt::AutoText);

	// misc

	#ifdef KS_PURE_QT
	static void about(QWidget *parent);
	#endif // KS_PURE_QT
	static void largeWidget(QWidget *parent, QWidget *widget, const QString &windowTitle, const QList<QPushButton *> &actionButtonList);
	static void plainText(QWidget *parent, const QString &text, const QString &windowTitle);
	static void systemInfo(QWidget *parent);

private:
	Q_DISABLE_COPY(UDialog)
	InfoWidget *m_infoWidget;
	QDialogButtonBox *m_buttonBox;
	QHBoxLayout *m_bottomLayout;
	QPushButton *m_acceptButton;
	QVBoxLayout *m_mainLayout;
	QWidget *m_bottomWidget;
	static QString makeHTMLLink(const QString &pathString);
};

class UMessageBuilder final {
public:
	enum class Type { Error, Info, Question, Warning };
	explicit UMessageBuilder(const Type value)
		: m_type(value) {
	}
	virtual ~UMessageBuilder() = default;

	bool exec(QWidget *parent);
	void showKey(const QString &key) {
		// NOTE: the group name is compatible with KF
		m_showVar = std::make_shared<Var>("Notification Messages", key, true);
	}
	void text(const QString &value, const Qt::TextFormat format) {
		m_text = value;
		m_textFormat = format;
	}

	void cancelDefault(const bool value) { m_cancelDefault = value; }
	void okText(const QString &value) { m_okText = value; }
	void icon(const QIcon &value) { m_icon = value; }
	void markdown(const QString &value) { text(value, Qt::MarkdownText); }
	void plainText(const QString &value) { text(value, Qt::PlainText); }
	void title(const QString &value) { m_title = value; }
private:
	bool m_cancelDefault = false;
	QIcon m_icon = QIcon();
	QString m_okText = QString();
	QString m_text = QString();
	QString m_title = QString();
	Qt::TextFormat m_textFormat = Qt::AutoText;
	std::shared_ptr<Var> m_showVar;
	Type m_type;
};
