// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A8__F0

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_16_0_F_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1153)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x1a43414bu, 0x3a7e086au, 40, 46 }, // 1a43414b3a7e086a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1153
    { 0x98fbacefu, 0xe2d64d8fu, 40, 63 }, // 98fbacefe2d64d8f = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1153
    { 0x56613a7bu, 0xc9584812u, 40, 6 }, // 56613a7bc9584812 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1153
    { 0x357ee48fu, 0x70364d03u, 40, 23 }, // 357ee48f70364d03 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1153
    { 0x09da3a46u, 0x6473402du, 97, 138 }, // 09da3a466473402d = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__32_32__CO__wave1_warp4_stg1--Arch_gfx1153
    { 0xc1c18affu, 0xcdd23762u, 120, 80 }, // c1c18affcdd23762 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1153
    { 0x3af53396u, 0x06506f0cu, 120, 155 }, // 3af5339606506f0c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1153
    { 0x5e7014e7u, 0x3082e329u, 120, 103 }, // 5e7014e73082e329 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1153
    { 0x67caae81u, 0x3e31554eu, 126, 138 }, // 67caae813e31554e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_32__CO__wave1_warp4_stg1--Arch_gfx1153
    { 0xd235e5cau, 0x9caf0743u, 126, 80 }, // d235e5ca9caf0743 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1153
    { 0xec7d9c11u, 0x47648b08u, 126, 155 }, // ec7d9c1147648b08 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_32__CO__wave3_warp4_stg1--Arch_gfx1153
    { 0x2daea578u, 0x124db30au, 126, 103 }, // 2daea578124db30a = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1153
    { 0xcbbafd21u, 0xa6372a23u, 132, 138 }, // cbbafd21a6372a23 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_64__CO__wave1_warp4_stg1--Arch_gfx1153
    { 0x11c97fd5u, 0xf47d43feu, 132, 80 }, // 11c97fd5f47d43fe = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_64__CO__wave2_warp4_stg1--Arch_gfx1153
    { 0xf7aa0b39u, 0xa7759c14u, 132, 155 }, // f7aa0b39a7759c14 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_64__CO__wave3_warp4_stg1--Arch_gfx1153
    { 0xc3251618u, 0xcd853700u, 132, 103 }, // c3251618cd853700 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊fp16@16_16_0_F_F_0___gfx11xx__P__64_64__CO__wave4_warp4_stg1--Arch_gfx1153
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1153_mod0
{{ 1, 4,14,14,15,15,14,14,12,14},
 { 7, 8,15,14,13,14,13,12,15,13},
 { 3, 1,10,14,15,15,13,14,13,14},
 { 1, 1, 8,11,11, 8,10, 9,10, 9},
 { 0, 0, 3, 8, 8, 8,15,12,15,15},
 { 1, 1, 3, 5, 0, 0, 9,10,13,14},
 { 6, 0, 5, 2, 2, 2, 2, 1, 5, 7},
 { 1, 6, 3, 1, 0, 2, 0, 3, 5,11},
 { 5, 1, 3, 0, 1, 0, 2, 0, 5,15},
 { 3, 2, 1, 1, 1, 0, 1, 6, 6, 5}}
// End of GPU gfx1153_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A8__F0 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 0;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 0;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

