/* Copyright (C) 2020 Google, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include "etnaviv-isa.h"

#include <stdint.h>
#include <util/bitset.h>

#define BITMASK_WORDS BITSET_WORDS(128)

typedef struct {
    BITSET_WORD bitset[BITMASK_WORDS];
} bitmask_t;


#define BITSET_FORMAT "08x%08x%08x%08x"
#define BITSET_VALUE(v) v[3], v[2], v[1], v[0]

static inline void
next_instruction(bitmask_t *instr, BITSET_WORD *start)
{
    instr->bitset[0] = *(start + 0);
    instr->bitset[1] = *(start + 1);
    instr->bitset[2] = *(start + 2);
    instr->bitset[3] = *(start + 3);
}

static inline uint64_t
bitmask_to_uint64_t(bitmask_t mask)
{
    return ((uint64_t)mask.bitset[1] << 32) | mask.bitset[0];
}

static inline bitmask_t
uint64_t_to_bitmask(uint64_t val)
{
    bitmask_t mask = {
        .bitset[0] = val & 0xffffffff,
        .bitset[1] = (val >> 32) & 0xffffffff,
    };

    return mask;
}

#include "isaspec_decode_decl.h"

static uint64_t
isa_decode_field(struct decode_scope *scope, const char *field_name);

/*
 * enum tables, these don't have any link back to other tables so just
 * dump them up front before the bitset tables
 */

static const struct isa_enum enum___cond = {
    .num_values = 17,
    .values = {
        { .val = 0, .display = "" },
        { .val = 1, .display = ".gt" },
        { .val = 2, .display = ".lt" },
        { .val = 3, .display = ".ge" },
        { .val = 4, .display = ".le" },
        { .val = 5, .display = ".eq" },
        { .val = 6, .display = ".ne" },
        { .val = 7, .display = ".and" },
        { .val = 8, .display = ".or" },
        { .val = 9, .display = ".xor" },
        { .val = 10, .display = ".not" },
        { .val = 11, .display = ".nz" },
        { .val = 12, .display = ".gez" },
        { .val = 13, .display = ".gz" },
        { .val = 14, .display = ".lez" },
        { .val = 15, .display = ".lz" },
        { .val = 22, .display = ".selmsb" },
    },
};
static const struct isa_enum enum___swiz = {
    .num_values = 4,
    .values = {
        { .val = 0, .display = "x" },
        { .val = 1, .display = "y" },
        { .val = 2, .display = "z" },
        { .val = 3, .display = "w" },
    },
};
static const struct isa_enum enum___type = {
    .num_values = 8,
    .values = {
        { .val = 0, .display = "" },
        { .val = 1, .display = ".s32" },
        { .val = 2, .display = ".s8" },
        { .val = 3, .display = ".u16" },
        { .val = 4, .display = ".f16" },
        { .val = 5, .display = ".s16" },
        { .val = 6, .display = ".u32" },
        { .val = 7, .display = ".u8" },
    },
};
static const struct isa_enum enum___reg_group = {
    .num_values = 6,
    .values = {
        { .val = 0, .display = "t" },
        { .val = 1, .display = "i" },
        { .val = 2, .display = "u" },
        { .val = 3, .display = "u" },
        { .val = 4, .display = "th" },
        { .val = 7, .display = "" },
    },
};
static const struct isa_enum enum___reg_addressing_mode = {
    .num_values = 5,
    .values = {
        { .val = 0, .display = "" },
        { .val = 1, .display = "[a.x]" },
        { .val = 2, .display = "[a.y]" },
        { .val = 3, .display = "[a.z]" },
        { .val = 4, .display = "[a.w]" },
    },
};
static const struct isa_enum enum___rounding = {
    .num_values = 5,
    .values = {
        { .val = 0, .display = "" },
        { .val = 1, .display = ".rtz" },
        { .val = 2, .display = ".rtne" },
        { .val = 3, .display = ".unkown" },
        { .val = 4, .display = ".unkown2" },
    },
};
static const struct isa_enum enum___wrmask = {
    .num_values = 16,
    .values = {
        { .val = 0, .display = ".____" },
        { .val = 1, .display = ".x___" },
        { .val = 2, .display = "._y__" },
        { .val = 3, .display = ".xy__" },
        { .val = 4, .display = ".__z_" },
        { .val = 5, .display = ".zx" },
        { .val = 6, .display = ".zy" },
        { .val = 7, .display = ".xyz_" },
        { .val = 8, .display = ".___w" },
        { .val = 9, .display = ".x__w" },
        { .val = 10, .display = "._y_w" },
        { .val = 11, .display = ".xy_w" },
        { .val = 12, .display = ".__zw" },
        { .val = 13, .display = ".x_zw" },
        { .val = 14, .display = "._yzw" },
        { .val = 15, .display = "" },
    },
};
static const struct isa_enum enum___thread = {
    .num_values = 3,
    .values = {
        { .val = 0, .display = "" },
        { .val = 1, .display = ".t0" },
        { .val = 2, .display = ".t1" },
    },
};

/*
 * generated expression functions, can be linked from bitset tables, so
 * also dump them up front
 */

static uint64_t
expr___instruction_has_src0_src1(struct decode_scope *scope)
{
    int64_t SRC0_USE = isa_decode_field(scope, "SRC0_USE");
    int64_t SRC1_USE = isa_decode_field(scope, "SRC1_USE");
    return (SRC0_USE != 0) && (SRC1_USE != 0);
}
static uint64_t
expr___instruction_has_src2(struct decode_scope *scope)
{
    int64_t SRC2_USE = isa_decode_field(scope, "SRC2_USE");
    return (SRC2_USE != 0);
}
static uint64_t
expr___instruction_has_src0_src1_src2(struct decode_scope *scope)
{
    int64_t SRC0_USE = isa_decode_field(scope, "SRC0_USE");
    int64_t SRC1_USE = isa_decode_field(scope, "SRC1_USE");
    int64_t SRC2_USE = isa_decode_field(scope, "SRC2_USE");
    return (SRC0_USE != 0) && (SRC1_USE != 0)  && (SRC2_USE != 0);
}
static uint64_t
expr___instruction_has_src0(struct decode_scope *scope)
{
    int64_t SRC0_USE = isa_decode_field(scope, "SRC0_USE");
    int64_t SRC1_USE = isa_decode_field(scope, "SRC1_USE");
    return (SRC0_USE != 0) && (SRC1_USE == 0);
}
static uint64_t
expr_anon_0(struct decode_scope *scope)
{
    int64_t DST_USE = isa_decode_field(scope, "DST_USE");
    return DST_USE == 0;
}
static uint64_t
expr_anon_1(struct decode_scope *scope)
{
    int64_t TYPE_BIT01 = isa_decode_field(scope, "TYPE_BIT01");
    int64_t TYPE_BIT2 = isa_decode_field(scope, "TYPE_BIT2");
    return TYPE_BIT2 << 2 | TYPE_BIT01;
}
static uint64_t
expr_anon_2(struct decode_scope *scope)
{
    int64_t HIGH_HALF = isa_decode_field(scope, "HIGH_HALF");
    int64_t LOW_HALF = isa_decode_field(scope, "LOW_HALF");
    return HIGH_HALF << 1 | LOW_HALF;
}
static uint64_t
expr_anon_3(struct decode_scope *scope)
{
    int64_t SRC_AMODE = isa_decode_field(scope, "SRC_AMODE");
    return (SRC_AMODE >> 1);
}
static uint64_t
expr_anon_4(struct decode_scope *scope)
{
    int64_t IMMED_TYPE = isa_decode_field(scope, "IMMED_TYPE");
    int64_t SRC_RGROUP = isa_decode_field(scope, "SRC_RGROUP");
    return ((SRC_RGROUP == 7) & (IMMED_TYPE == 0));
}
static uint64_t
expr_anon_5(struct decode_scope *scope)
{
    int64_t SRC_ABS = isa_decode_field(scope, "SRC_ABS");
    int64_t SRC_AMODE = isa_decode_field(scope, "SRC_AMODE");
    int64_t SRC_NEG = isa_decode_field(scope, "SRC_NEG");
    int64_t SRC_REG = isa_decode_field(scope, "SRC_REG");
    int64_t SRC_SWIZ = isa_decode_field(scope, "SRC_SWIZ");
    return ((((SRC_AMODE & 0x1) << 19) |
				   (SRC_ABS << 18) |
				   (SRC_NEG << 17) |
				   (SRC_SWIZ << 9) |
				   SRC_REG) << 12);
}
static uint64_t
expr_anon_6(struct decode_scope *scope)
{
    int64_t IMMED_TYPE = isa_decode_field(scope, "IMMED_TYPE");
    int64_t SRC_RGROUP = isa_decode_field(scope, "SRC_RGROUP");
    return ((SRC_RGROUP == 7) & (IMMED_TYPE == 1));
}
static uint64_t
expr_anon_7(struct decode_scope *scope)
{
    int64_t SRC_ABS = isa_decode_field(scope, "SRC_ABS");
    int64_t SRC_AMODE = isa_decode_field(scope, "SRC_AMODE");
    int64_t SRC_NEG = isa_decode_field(scope, "SRC_NEG");
    int64_t SRC_REG = isa_decode_field(scope, "SRC_REG");
    int64_t SRC_SWIZ = isa_decode_field(scope, "SRC_SWIZ");
    return (((SRC_AMODE & 0x1) << 19) |
				  (SRC_ABS << 18) |
				  (SRC_NEG << 17) |
				  (SRC_SWIZ << 9) |
				   SRC_REG);
}
static uint64_t
expr_anon_8(struct decode_scope *scope)
{
    int64_t IMMED_TYPE = isa_decode_field(scope, "IMMED_TYPE");
    int64_t SRC_RGROUP = isa_decode_field(scope, "SRC_RGROUP");
    return ((SRC_RGROUP == 7) & (IMMED_TYPE == 2));
}
static uint64_t
expr_anon_9(struct decode_scope *scope)
{
    int64_t SRC_ABS = isa_decode_field(scope, "SRC_ABS");
    int64_t SRC_AMODE = isa_decode_field(scope, "SRC_AMODE");
    int64_t SRC_NEG = isa_decode_field(scope, "SRC_NEG");
    int64_t SRC_REG = isa_decode_field(scope, "SRC_REG");
    int64_t SRC_SWIZ = isa_decode_field(scope, "SRC_SWIZ");
    return (((SRC_AMODE & 0x1) << 19) |
				  (SRC_ABS << 18) |
				  (SRC_NEG << 17) |
				  (SRC_SWIZ << 9) |
				   SRC_REG);
}
static uint64_t
expr_anon_10(struct decode_scope *scope)
{
    int64_t IMMED_TYPE = isa_decode_field(scope, "IMMED_TYPE");
    int64_t SRC_RGROUP = isa_decode_field(scope, "SRC_RGROUP");
    return ((SRC_RGROUP == 7) & (IMMED_TYPE == 3));
}
static uint64_t
expr_anon_11(struct decode_scope *scope)
{
    int64_t SRC_ABS = isa_decode_field(scope, "SRC_ABS");
    int64_t SRC_AMODE = isa_decode_field(scope, "SRC_AMODE");
    int64_t SRC_NEG = isa_decode_field(scope, "SRC_NEG");
    int64_t SRC_REG = isa_decode_field(scope, "SRC_REG");
    int64_t SRC_SWIZ = isa_decode_field(scope, "SRC_SWIZ");
    return (((SRC_AMODE & 0x1) << 19) |
				  (SRC_ABS << 18) |
				  (SRC_NEG << 17) |
				  (SRC_SWIZ << 9) |
				   SRC_REG);
}
static uint64_t
expr_anon_12(struct decode_scope *scope)
{
    int64_t HIGH_HALF = isa_decode_field(scope, "HIGH_HALF");
    int64_t LOW_HALF = isa_decode_field(scope, "LOW_HALF");
    return HIGH_HALF << 1 | LOW_HALF;
}
static uint64_t
expr_anon_13(struct decode_scope *scope)
{
    int64_t LEFT_SHIFT = isa_decode_field(scope, "LEFT_SHIFT");
    return LEFT_SHIFT != 0;
}
static uint64_t
expr_anon_14(struct decode_scope *scope)
{
    int64_t HIGH_HALF = isa_decode_field(scope, "HIGH_HALF");
    int64_t LOW_HALF = isa_decode_field(scope, "LOW_HALF");
    return HIGH_HALF << 1 | LOW_HALF;
}
static uint64_t
expr_anon_15(struct decode_scope *scope)
{
    int64_t HIGH_HALF = isa_decode_field(scope, "HIGH_HALF");
    int64_t LOW_HALF = isa_decode_field(scope, "LOW_HALF");
    return HIGH_HALF << 1 | LOW_HALF;
}

/* forward-declarations of bitset decode functions */
static const struct isa_field_decode decode___instruction_dst_gen_0_fields[] = {
};
static void decode___instruction_dst_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_gen_0_fields[] = {
};
static void decode___instruction_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___src_swizzle_gen_0_fields[] = {
};
static void decode___src_swizzle_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_gen_0_fields[] = {
};
static void decode___instruction_alu_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_src_gen_0_fields[] = {
};
static void decode___instruction_src_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_no_src_gen_0_fields[] = {
};
static void decode___instruction_alu_no_src_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_no_dst_maybe_src0_src1_gen_0_fields[] = {
};
static void decode___instruction_alu_no_dst_maybe_src0_src1_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_src0_gen_0_fields[] = {
};
static void decode___instruction_alu_src0_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_src2_gen_0_fields[] = {
};
static void decode___instruction_alu_src2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_src0_src1_gen_0_fields[] = {
};
static void decode___instruction_alu_src0_src1_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_src0_src2_gen_0_fields[] = {
};
static void decode___instruction_alu_src0_src2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_src1_src2_gen_0_fields[] = {
};
static void decode___instruction_alu_src1_src2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_alu_src0_src1_src2_gen_0_fields[] = {
};
static void decode___instruction_alu_src0_src1_src2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_tex_gen_0_fields[] = {
};
static void decode___instruction_tex_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_tex_src0_maybe_src2_gen_0_fields[] = {
};
static void decode___instruction_tex_src0_maybe_src2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_tex_src0_src1_gen_0_fields[] = {
};
static void decode___instruction_tex_src0_src1_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_tex_src0_src1_src2_gen_0_fields[] = {
};
static void decode___instruction_tex_src0_src1_src2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_tex_src0_maybe_src1_src2_gen_0_fields[] = {
};
static void decode___instruction_tex_src0_maybe_src1_src2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_cf_gen_0_fields[] = {
};
static void decode___instruction_cf_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_cf_no_src_gen_0_fields[] = {
};
static void decode___instruction_cf_no_src_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_cf_src0_gen_0_fields[] = {
};
static void decode___instruction_cf_src0_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_cf_src0_src1_gen_0_fields[] = {
};
static void decode___instruction_cf_src0_src1_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___left_shift_gen_0_fields[] = {
};
static void decode___left_shift_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_load_gen_0_fields[] = {
};
static void decode___instruction_load_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___instruction_store_gen_0_fields[] = {
};
static void decode___instruction_store_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_nop_gen_0_fields[] = {
};
static void decode_nop_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_add_gen_0_fields[] = {
};
static void decode_add_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_mad_gen_0_fields[] = {
};
static void decode_mad_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_mul_gen_0_fields[] = {
};
static void decode_mul_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_dp3_gen_0_fields[] = {
};
static void decode_dp3_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_dp4_gen_0_fields[] = {
};
static void decode_dp4_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_dsx_gen_0_fields[] = {
};
static void decode_dsx_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_dsy_gen_0_fields[] = {
};
static void decode_dsy_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_mov_gen_0_fields[] = {
};
static void decode_mov_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_movar_gen_0_fields[] = {
};
static void decode_movar_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_rcp_gen_0_fields[] = {
};
static void decode_rcp_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_rsq_gen_0_fields[] = {
};
static void decode_rsq_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_select_gen_0_fields[] = {
};
static void decode_select_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_set_gen_0_fields[] = {
};
static void decode_set_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_exp_gen_0_fields[] = {
};
static void decode_exp_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_log_gen_0_fields[] = {
};
static void decode_log_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_frc_gen_0_fields[] = {
};
static void decode_frc_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_call_gen_0_fields[] = {
};
static void decode_call_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_ret_gen_0_fields[] = {
};
static void decode_ret_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_branch_gen_0_fields[] = {
};
static void decode_branch_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_branch_unary_gen_0_fields[] = {
};
static void decode_branch_unary_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_branch_binary_gen_0_fields[] = {
};
static void decode_branch_binary_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_texkill_gen_0_fields[] = {
};
static void decode_texkill_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_texld_gen_0_fields[] = {
};
static void decode_texld_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_texldb_gen_0_fields[] = {
};
static void decode_texldb_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_texldd_gen_0_fields[] = {
};
static void decode_texldd_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_texldl_gen_0_fields[] = {
};
static void decode_texldl_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_sqrt_gen_0_fields[] = {
};
static void decode_sqrt_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_sin_gen_0_fields[] = {
};
static void decode_sin_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_cos_gen_0_fields[] = {
};
static void decode_cos_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_branch_any_gen_0_fields[] = {
};
static void decode_branch_any_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_floor_gen_0_fields[] = {
};
static void decode_floor_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_ceil_gen_0_fields[] = {
};
static void decode_ceil_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_sign_gen_0_fields[] = {
};
static void decode_sign_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_barrier_gen_0_fields[] = {
};
static void decode_barrier_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_swizzle_gen_0_fields[] = {
};
static void decode_swizzle_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_i2i_gen_0_fields[] = {
};
static void decode_i2i_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_i2f_gen_0_fields[] = {
};
static void decode_i2f_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_f2i_gen_0_fields[] = {
};
static void decode_f2i_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_f2irnd_gen_0_fields[] = {
};
static void decode_f2irnd_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_cmp_gen_0_fields[] = {
};
static void decode_cmp_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_load_gen_0_fields[] = {
};
static void decode_load_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_store_gen_0_fields[] = {
};
static void decode_store_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_clamp0_max_gen_0_fields[] = {
};
static void decode_clamp0_max_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_iaddsat_gen_0_fields[] = {
};
static void decode_iaddsat_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_imullo0_gen_0_fields[] = {
};
static void decode_imullo0_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_imulhi0_gen_0_fields[] = {
};
static void decode_imulhi0_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_idiv0_gen_0_fields[] = {
};
static void decode_idiv0_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_imod_gen_0_fields[] = {
};
static void decode_imod_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_txf_gen_0_fields[] = {
};
static void decode_txf_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_imadlo0_gen_0_fields[] = {
};
static void decode_imadlo0_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_imadlosat0_gen_0_fields[] = {
};
static void decode_imadlosat0_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_movai_gen_0_fields[] = {
};
static void decode_movai_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_iabs_gen_0_fields[] = {
};
static void decode_iabs_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_leadzero_gen_0_fields[] = {
};
static void decode_leadzero_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_lshift_gen_0_fields[] = {
};
static void decode_lshift_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_rshift_gen_0_fields[] = {
};
static void decode_rshift_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_rotate_gen_0_fields[] = {
};
static void decode_rotate_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_or_gen_0_fields[] = {
};
static void decode_or_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_and_gen_0_fields[] = {
};
static void decode_and_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_xor_gen_0_fields[] = {
};
static void decode_xor_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_not_gen_0_fields[] = {
};
static void decode_not_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_bit_extract_gen_0_fields[] = {
};
static void decode_bit_extract_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_popcount_gen_0_fields[] = {
};
static void decode_popcount_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_div_gen_0_fields[] = {
};
static void decode_div_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_atomic_add_gen_0_fields[] = {
};
static void decode_atomic_add_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_atomic_xchg_gen_0_fields[] = {
};
static void decode_atomic_xchg_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_atomic_cmp_xchg_gen_0_fields[] = {
};
static void decode_atomic_cmp_xchg_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_atomic_min_gen_0_fields[] = {
};
static void decode_atomic_min_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_atomic_max_gen_0_fields[] = {
};
static void decode_atomic_max_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_atomic_or_gen_0_fields[] = {
};
static void decode_atomic_or_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_atomic_and_gen_0_fields[] = {
};
static void decode_atomic_and_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_atomic_xor_gen_0_fields[] = {
};
static void decode_atomic_xor_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_bit_rev_gen_0_fields[] = {
};
static void decode_bit_rev_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_texldlpcf_gen_0_fields[] = {
};
static void decode_texldlpcf_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_texldgpcf_gen_0_fields[] = {
};
static void decode_texldgpcf_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_conv_gen_0_fields[] = {
};
static void decode_conv_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_dp2_gen_0_fields[] = {
};
static void decode_dp2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_norm_dp2_gen_0_fields[] = {
};
static void decode_norm_dp2_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_norm_dp3_gen_0_fields[] = {
};
static void decode_norm_dp3_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_norm_dp4_gen_0_fields[] = {
};
static void decode_norm_dp4_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_norm_mul_gen_0_fields[] = {
};
static void decode_norm_mul_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_img_load_gen_0_fields[] = {
};
static void decode_img_load_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_img_store_gen_0_fields[] = {
};
static void decode_img_store_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_tg4_gen_0_fields[] = {
};
static void decode_tg4_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___extended_instruction_alu_gen_0_fields[] = {
};
static void decode___extended_instruction_alu_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode___extended_instruction_alu_src0_gen_0_fields[] = {
};
static void decode___extended_instruction_alu_src0_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_bit_findlsb_gen_0_fields[] = {
};
static void decode_bit_findlsb_gen_0(void *out, struct decode_scope *scope);
static const struct isa_field_decode decode_bit_findmsb_gen_0_fields[] = {
};
static void decode_bit_findmsb_gen_0(void *out, struct decode_scope *scope);

/*
 * Forward-declarations (so we don't have to figure out which order to
 * emit various tables when they have pointers to each other)
 */

static const struct isa_bitset bitset___instruction_dst_gen_0;
static const struct isa_bitset bitset___instruction_gen_0;
static const struct isa_bitset bitset___src_swizzle_gen_0;
static const struct isa_bitset bitset___instruction_alu_gen_0;
static const struct isa_bitset bitset___instruction_src_gen_0;
static const struct isa_bitset bitset___instruction_alu_no_src_gen_0;
static const struct isa_bitset bitset___instruction_alu_no_dst_maybe_src0_src1_gen_0;
static const struct isa_bitset bitset___instruction_alu_src0_gen_0;
static const struct isa_bitset bitset___instruction_alu_src2_gen_0;
static const struct isa_bitset bitset___instruction_alu_src0_src1_gen_0;
static const struct isa_bitset bitset___instruction_alu_src0_src2_gen_0;
static const struct isa_bitset bitset___instruction_alu_src1_src2_gen_0;
static const struct isa_bitset bitset___instruction_alu_src0_src1_src2_gen_0;
static const struct isa_bitset bitset___instruction_tex_gen_0;
static const struct isa_bitset bitset___instruction_tex_src0_maybe_src2_gen_0;
static const struct isa_bitset bitset___instruction_tex_src0_src1_gen_0;
static const struct isa_bitset bitset___instruction_tex_src0_src1_src2_gen_0;
static const struct isa_bitset bitset___instruction_tex_src0_maybe_src1_src2_gen_0;
static const struct isa_bitset bitset___instruction_cf_gen_0;
static const struct isa_bitset bitset___instruction_cf_no_src_gen_0;
static const struct isa_bitset bitset___instruction_cf_src0_gen_0;
static const struct isa_bitset bitset___instruction_cf_src0_src1_gen_0;
static const struct isa_bitset bitset___left_shift_gen_0;
static const struct isa_bitset bitset___instruction_load_gen_0;
static const struct isa_bitset bitset___instruction_store_gen_0;
static const struct isa_bitset bitset_nop_gen_0;
static const struct isa_bitset bitset_add_gen_0;
static const struct isa_bitset bitset_mad_gen_0;
static const struct isa_bitset bitset_mul_gen_0;
static const struct isa_bitset bitset_dp3_gen_0;
static const struct isa_bitset bitset_dp4_gen_0;
static const struct isa_bitset bitset_dsx_gen_0;
static const struct isa_bitset bitset_dsy_gen_0;
static const struct isa_bitset bitset_mov_gen_0;
static const struct isa_bitset bitset_movar_gen_0;
static const struct isa_bitset bitset_rcp_gen_0;
static const struct isa_bitset bitset_rsq_gen_0;
static const struct isa_bitset bitset_select_gen_0;
static const struct isa_bitset bitset_set_gen_0;
static const struct isa_bitset bitset_exp_gen_0;
static const struct isa_bitset bitset_log_gen_0;
static const struct isa_bitset bitset_frc_gen_0;
static const struct isa_bitset bitset_call_gen_0;
static const struct isa_bitset bitset_ret_gen_0;
static const struct isa_bitset bitset_branch_gen_0;
static const struct isa_bitset bitset_branch_unary_gen_0;
static const struct isa_bitset bitset_branch_binary_gen_0;
static const struct isa_bitset bitset_texkill_gen_0;
static const struct isa_bitset bitset_texld_gen_0;
static const struct isa_bitset bitset_texldb_gen_0;
static const struct isa_bitset bitset_texldd_gen_0;
static const struct isa_bitset bitset_texldl_gen_0;
static const struct isa_bitset bitset_sqrt_gen_0;
static const struct isa_bitset bitset_sin_gen_0;
static const struct isa_bitset bitset_cos_gen_0;
static const struct isa_bitset bitset_branch_any_gen_0;
static const struct isa_bitset bitset_floor_gen_0;
static const struct isa_bitset bitset_ceil_gen_0;
static const struct isa_bitset bitset_sign_gen_0;
static const struct isa_bitset bitset_barrier_gen_0;
static const struct isa_bitset bitset_swizzle_gen_0;
static const struct isa_bitset bitset_i2i_gen_0;
static const struct isa_bitset bitset_i2f_gen_0;
static const struct isa_bitset bitset_f2i_gen_0;
static const struct isa_bitset bitset_f2irnd_gen_0;
static const struct isa_bitset bitset_cmp_gen_0;
static const struct isa_bitset bitset_load_gen_0;
static const struct isa_bitset bitset_store_gen_0;
static const struct isa_bitset bitset_clamp0_max_gen_0;
static const struct isa_bitset bitset_iaddsat_gen_0;
static const struct isa_bitset bitset_imullo0_gen_0;
static const struct isa_bitset bitset_imulhi0_gen_0;
static const struct isa_bitset bitset_idiv0_gen_0;
static const struct isa_bitset bitset_imod_gen_0;
static const struct isa_bitset bitset_txf_gen_0;
static const struct isa_bitset bitset_imadlo0_gen_0;
static const struct isa_bitset bitset_imadlosat0_gen_0;
static const struct isa_bitset bitset_movai_gen_0;
static const struct isa_bitset bitset_iabs_gen_0;
static const struct isa_bitset bitset_leadzero_gen_0;
static const struct isa_bitset bitset_lshift_gen_0;
static const struct isa_bitset bitset_rshift_gen_0;
static const struct isa_bitset bitset_rotate_gen_0;
static const struct isa_bitset bitset_or_gen_0;
static const struct isa_bitset bitset_and_gen_0;
static const struct isa_bitset bitset_xor_gen_0;
static const struct isa_bitset bitset_not_gen_0;
static const struct isa_bitset bitset_bit_extract_gen_0;
static const struct isa_bitset bitset_popcount_gen_0;
static const struct isa_bitset bitset_div_gen_0;
static const struct isa_bitset bitset_atomic_add_gen_0;
static const struct isa_bitset bitset_atomic_xchg_gen_0;
static const struct isa_bitset bitset_atomic_cmp_xchg_gen_0;
static const struct isa_bitset bitset_atomic_min_gen_0;
static const struct isa_bitset bitset_atomic_max_gen_0;
static const struct isa_bitset bitset_atomic_or_gen_0;
static const struct isa_bitset bitset_atomic_and_gen_0;
static const struct isa_bitset bitset_atomic_xor_gen_0;
static const struct isa_bitset bitset_bit_rev_gen_0;
static const struct isa_bitset bitset_texldlpcf_gen_0;
static const struct isa_bitset bitset_texldgpcf_gen_0;
static const struct isa_bitset bitset_conv_gen_0;
static const struct isa_bitset bitset_dp2_gen_0;
static const struct isa_bitset bitset_norm_dp2_gen_0;
static const struct isa_bitset bitset_norm_dp3_gen_0;
static const struct isa_bitset bitset_norm_dp4_gen_0;
static const struct isa_bitset bitset_norm_mul_gen_0;
static const struct isa_bitset bitset_img_load_gen_0;
static const struct isa_bitset bitset_img_store_gen_0;
static const struct isa_bitset bitset_tg4_gen_0;
static const struct isa_bitset bitset___extended_instruction_alu_gen_0;
static const struct isa_bitset bitset___extended_instruction_alu_src0_gen_0;
static const struct isa_bitset bitset_bit_findlsb_gen_0;
static const struct isa_bitset bitset_bit_findmsb_gen_0;

static const struct isa_bitset *__instruction_dst[];
static const struct isa_bitset *__instruction[];
static const struct isa_bitset *__src_swizzle[];
static const struct isa_bitset *__instruction_src[];
static const struct isa_bitset *__left_shift[];

/*
 * bitset tables:
 */

static const struct isa_case __instruction_dst__case0_gen_0 = {
       .expr     = &expr_anon_0,
       .display  = "void",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_case __instruction_dst__default_gen_0 = {
       .display  = "t{REG}{AMODE}{COMPS}",
       .num_fields = 3,
       .fields   = {
          { .name = "AMODE", .low = 0, .high = 2,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "REG", .low = 3, .high = 9,
            .type = TYPE_UINT,
          },
          { .name = "COMPS", .low = 10, .high = 13,
            .type = TYPE_ENUM,
            .enums = &enum___wrmask,
          },
       },
};
static const struct isa_bitset bitset___instruction_dst_gen_0 = {

       .name     = "#instruction-dst",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x0, 0x0, 0x0 },
       .decode = decode___instruction_dst_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_dst_gen_0_fields),
       .decode_fields = decode___instruction_dst_gen_0_fields,
       .num_cases = 2,
       .cases    = {
            &__instruction_dst__case0_gen_0,
            &__instruction_dst__default_gen_0,
       },
};
static const struct isa_case __instruction__default_gen_0 = {
       .num_fields = 5,
       .fields   = {
          { .name = "TYPE", .low = 0, .high = 0,
            .expr = &expr_anon_1,
            .type = TYPE_ENUM,
            .enums = &enum___type,
          },
          { .name = "COND", .low = 6, .high = 10,
            .type = TYPE_ENUM,
            .enums = &enum___cond,
          },
          { .name = "SAT", .low = 11, .high = 11,
            .display = ".sat",
            .type = TYPE_BOOL,
          },
          { .name = "TYPE_BIT2", .low = 53, .high = 53,
            .type = TYPE_UINT,
          },
          { .name = "TYPE_BIT01", .low = 94, .high = 95,
            .type = TYPE_INT,
          },
       },
};
static const struct isa_bitset bitset___instruction_gen_0 = {

       .name     = "#instruction",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x0, 0x0, 0x0 },
       .decode = decode___instruction_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_gen_0_fields),
       .decode_fields = decode___instruction_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction__default_gen_0,
       },
};
static const struct isa_case __src_swizzle__default_gen_0 = {
       .display  = ".{SWIZ_X}{SWIZ_Y}{SWIZ_Z}{SWIZ_W}",
       .num_fields = 4,
       .fields   = {
          { .name = "SWIZ_X", .low = 0, .high = 1,
            .type = TYPE_ENUM,
            .enums = &enum___swiz,
          },
          { .name = "SWIZ_Y", .low = 2, .high = 3,
            .type = TYPE_ENUM,
            .enums = &enum___swiz,
          },
          { .name = "SWIZ_Z", .low = 4, .high = 5,
            .type = TYPE_ENUM,
            .enums = &enum___swiz,
          },
          { .name = "SWIZ_W", .low = 6, .high = 7,
            .type = TYPE_ENUM,
            .enums = &enum___swiz,
          },
       },
};
static const struct isa_bitset bitset___src_swizzle_gen_0 = {

       .name     = "#src-swizzle",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x0, 0x0, 0x0 },
       .decode = decode___src_swizzle_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___src_swizzle_gen_0_fields),
       .decode_fields = decode___src_swizzle_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__src_swizzle__default_gen_0,
       },
};
static const struct isa_field_params __instruction_alu__default_gen_0_dst = {
       .num_params = 2,
       .params = {
           { .name= "DST_USE",  .as = "DST_USE" },
           { .name= "COMPS",  .as = "COMPS" },

       },
};
static const struct isa_case __instruction_alu__default_gen_0 = {
       .num_fields = 10,
       .fields   = {
          { .name = "THREAD", .low = 0, .high = 0,
            .expr = &expr_anon_2,
            .type = TYPE_ENUM,
            .enums = &enum___thread,
          },
          { .name = "DST_USE", .low = 12, .high = 12,
            .type = TYPE_BOOL,
          },
          { .name = "DST", .low = 13, .high = 26,
            .type = TYPE_BITSET,
            .bitsets = __instruction_dst,
            .params = &__instruction_alu__default_gen_0_dst,
          },
          { .name = "UNK", .low = 27, .high = 27,
            .display = ".unk",
            .type = TYPE_BOOL,
          },
          { .name = "RMODE", .low = 32, .high = 33,
            .type = TYPE_ENUM,
            .enums = &enum___rounding,
          },
          { .name = "PMODE", .low = 34, .high = 34,
            .display = ".pack",
            .type = TYPE_BOOL_INV,
          },
          { .name = "SKPHP", .low = 39, .high = 39,
            .display = ".skpHp",
            .type = TYPE_BOOL,
          },
          { .name = "LOW_HALF", .low = 109, .high = 109,
            .type = TYPE_BOOL,
          },
          { .name = "HIGH_HALF", .low = 120, .high = 120,
            .type = TYPE_BOOL,
          },
          { .name = "DST_FULL", .low = 127, .high = 127,
            .display = ".hp",
            .type = TYPE_BOOL,
          },
       },
};
static const struct isa_bitset bitset___instruction_alu_gen_0 = {

       .parent   = &bitset___instruction_gen_0,
       .name     = "#instruction-alu",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0x778, 0x0, 0x0 },
       .decode = decode___instruction_alu_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_gen_0_fields),
       .decode_fields = decode___instruction_alu_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_alu__default_gen_0,
       },
};
static const struct isa_case __instruction_src__case0_gen_0 = {
       .expr     = &expr_anon_4,
       .display  = "{IMMED}:f20",
       .num_fields = 1,
       .fields   = {
          { .name = "IMMED", .low = 0, .high = 31,
            .expr = &expr_anon_5,
            .type = TYPE_FLOAT,
          },
       },
};
static const struct isa_case __instruction_src__case1_gen_0 = {
       .expr     = &expr_anon_6,
       .display  = "{IMMED}:s20",
       .num_fields = 1,
       .fields   = {
          { .name = "IMMED", .low = 0, .high = 19,
            .expr = &expr_anon_7,
            .type = TYPE_INT,
          },
       },
};
static const struct isa_case __instruction_src__case2_gen_0 = {
       .expr     = &expr_anon_8,
       .display  = "{IMMED}:u20",
       .num_fields = 1,
       .fields   = {
          { .name = "IMMED", .low = 0, .high = 19,
            .expr = &expr_anon_9,
            .type = TYPE_UINT,
          },
       },
};
static const struct isa_case __instruction_src__case3_gen_0 = {
       .expr     = &expr_anon_10,
       .display  = "{IMMED}:f16",
       .num_fields = 1,
       .fields   = {
          { .name = "IMMED", .low = 0, .high = 15,
            .expr = &expr_anon_11,
            .type = TYPE_FLOAT,
          },
       },
};
static const struct isa_case __instruction_src__default_gen_0 = {
       .display  = "{SRC_NEG}{SRC_ABS}{SRC_RGROUP}{SRC_REG}{SRC_AMODE}{SRC_SWIZ}{SRC_ABS}",
       .num_fields = 4,
       .fields   = {
          { .name = "IMMED_TYPE", .low = 0, .high = 0,
            .expr = &expr_anon_3,
            .type = TYPE_UINT,
          },
          { .name = "SRC_SWIZ", .low = 0, .high = 7,
            .type = TYPE_BITSET,
            .bitsets = __src_swizzle,
          },
          { .name = "SRC_NEG", .low = 8, .high = 8,
            .display = "-",
            .type = TYPE_BOOL,
          },
          { .name = "SRC_ABS", .low = 9, .high = 9,
            .display = "|",
            .type = TYPE_BOOL,
          },
       },
};
static const struct isa_bitset bitset___instruction_src_gen_0 = {

       .name     = "#instruction-src",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x0, 0x0, 0x0 },
       .decode = decode___instruction_src_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_src_gen_0_fields),
       .decode_fields = decode___instruction_src_gen_0_fields,
       .num_cases = 5,
       .cases    = {
            &__instruction_src__case0_gen_0,
            &__instruction_src__case1_gen_0,
            &__instruction_src__case2_gen_0,
            &__instruction_src__case3_gen_0,
            &__instruction_src__default_gen_0,
       },
};
static const struct isa_case __instruction_alu_no_src__default_gen_0 = {
       .display  = "{NAME} {DST:align=18}{DST_FULL}, void, void, void",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset___instruction_alu_no_src_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#instruction-alu-no-src",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0xffdfff78, 0x3ffeffff, 0x7effdfff },
       .decode = decode___instruction_alu_no_src_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_no_src_gen_0_fields),
       .decode_fields = decode___instruction_alu_no_src_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_alu_no_src__default_gen_0,
       },
};
static const struct isa_case __instruction_alu_no_dst_maybe_src0_src1__case0_gen_0 = {
       .expr     = &expr___instruction_has_src0_src1,
       .display  = "{NAME}{COND}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, {SRC0}, {SRC1}, void",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_case __instruction_alu_no_dst_maybe_src0_src1__case1_gen_0 = {
       .expr     = &expr___instruction_has_src0,
       .display  = "{NAME}{COND}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, {SRC0}, void, void",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_field_params __instruction_alu_no_dst_maybe_src0_src1__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_alu_no_dst_maybe_src0_src1__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_alu_no_dst_maybe_src0_src1__default_gen_0 = {
       .display  = "{NAME}{COND}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}{DST_FULL}, void, void, void",
       .num_fields = 10,
       .fields   = {
          { .name = "SRC0_USE", .low = 43, .high = 43,
            .type = TYPE_BOOL,
          },
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_no_dst_maybe_src0_src1__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_USE", .low = 70, .high = 70,
            .type = TYPE_BOOL,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_no_dst_maybe_src0_src1__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_alu_no_dst_maybe_src0_src1_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#instruction-alu-no-dst-maybe-src0-src1",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0x778, 0x0, 0x7effdff8 },
       .decode = decode___instruction_alu_no_dst_maybe_src0_src1_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_no_dst_maybe_src0_src1_gen_0_fields),
       .decode_fields = decode___instruction_alu_no_dst_maybe_src0_src1_gen_0_fields,
       .num_cases = 3,
       .cases    = {
            &__instruction_alu_no_dst_maybe_src0_src1__case0_gen_0,
            &__instruction_alu_no_dst_maybe_src0_src1__case1_gen_0,
            &__instruction_alu_no_dst_maybe_src0_src1__default_gen_0,
       },
};
static const struct isa_field_params __instruction_alu_src0__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_alu_src0__default_gen_0 = {
       .display  = "{NAME}{DST_FULL}{SAT}{COND}{SKPHP}{UNK}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, {SRC0}, void, void",
       .num_fields = 4,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src0__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_alu_src0_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#instruction-alu-src0",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0xf78, 0x3ffeffc0, 0x7effdfff },
       .decode = decode___instruction_alu_src0_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_src0_gen_0_fields),
       .decode_fields = decode___instruction_alu_src0_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_alu_src0__default_gen_0,
       },
};
static const struct isa_field_params __instruction_alu_src2__default_gen_0_src2 = {
       .num_params = 3,
       .params = {
           { .name= "SRC2_REG",  .as = "SRC_REG" },
           { .name= "SRC2_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC2_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_alu_src2__default_gen_0 = {
       .display  = "{NAME}{DST_FULL}{SAT}{COND}{SKPHP}{UNK}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, void, void, {SRC2}",
       .num_fields = 4,
       .fields   = {
          { .name = "SRC2_REG", .low = 100, .high = 108,
            .type = TYPE_UINT,
          },
          { .name = "SRC2", .low = 110, .high = 119,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src2__default_gen_0_src2,
          },
          { .name = "SRC2_AMODE", .low = 121, .high = 123,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC2_RGROUP", .low = 124, .high = 126,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_alu_src2_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#instruction-alu-src2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0xffdfff78, 0x3ffeffff, 0xf },
       .decode = decode___instruction_alu_src2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_src2_gen_0_fields),
       .decode_fields = decode___instruction_alu_src2_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_alu_src2__default_gen_0,
       },
};
static const struct isa_field_params __instruction_alu_src0_src1__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_alu_src0_src1__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_alu_src0_src1__default_gen_0 = {
       .display  = "{NAME}{DST_FULL}{SAT}{COND}{SKPHP}{UNK}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, {SRC0}, {SRC1}, void",
       .num_fields = 8,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src0_src1__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src0_src1__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_alu_src0_src1_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#instruction-alu-src0-src1",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0xf78, 0x40, 0x7effdff8 },
       .decode = decode___instruction_alu_src0_src1_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_src0_src1_gen_0_fields),
       .decode_fields = decode___instruction_alu_src0_src1_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_alu_src0_src1__default_gen_0,
       },
};
static const struct isa_field_params __instruction_alu_src0_src2__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_alu_src0_src2__default_gen_0_src2 = {
       .num_params = 3,
       .params = {
           { .name= "SRC2_REG",  .as = "SRC_REG" },
           { .name= "SRC2_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC2_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_alu_src0_src2__default_gen_0 = {
       .display  = "{NAME}{DST_FULL}{SAT}{COND}{SKPHP}{UNK}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, {SRC0}, void, {SRC2}",
       .num_fields = 8,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src0_src2__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC2_REG", .low = 100, .high = 108,
            .type = TYPE_UINT,
          },
          { .name = "SRC2", .low = 110, .high = 119,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src0_src2__default_gen_0_src2,
          },
          { .name = "SRC2_AMODE", .low = 121, .high = 123,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC2_RGROUP", .low = 124, .high = 126,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_alu_src0_src2_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#instruction-alu-src0-src2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0xf78, 0x3ffeffc0, 0xf },
       .decode = decode___instruction_alu_src0_src2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_src0_src2_gen_0_fields),
       .decode_fields = decode___instruction_alu_src0_src2_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_alu_src0_src2__default_gen_0,
       },
};
static const struct isa_field_params __instruction_alu_src1_src2__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_alu_src1_src2__default_gen_0_src2 = {
       .num_params = 3,
       .params = {
           { .name= "SRC2_REG",  .as = "SRC_REG" },
           { .name= "SRC2_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC2_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_alu_src1_src2__default_gen_0 = {
       .display  = "{NAME}{DST_FULL}{SAT}{COND}{SKPHP}{UNK}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, void, {SRC1}, {SRC2}",
       .num_fields = 8,
       .fields   = {
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src1_src2__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC2_REG", .low = 100, .high = 108,
            .type = TYPE_UINT,
          },
          { .name = "SRC2", .low = 110, .high = 119,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src1_src2__default_gen_0_src2,
          },
          { .name = "SRC2_AMODE", .low = 121, .high = 123,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC2_RGROUP", .low = 124, .high = 126,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_alu_src1_src2_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#instruction-alu-src1-src2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x40, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0xffdfff78, 0x7f, 0x8 },
       .decode = decode___instruction_alu_src1_src2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_src1_src2_gen_0_fields),
       .decode_fields = decode___instruction_alu_src1_src2_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_alu_src1_src2__default_gen_0,
       },
};
static const struct isa_field_params __instruction_alu_src0_src1_src2__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_alu_src0_src1_src2__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_alu_src0_src1_src2__default_gen_0_src2 = {
       .num_params = 3,
       .params = {
           { .name= "SRC2_REG",  .as = "SRC_REG" },
           { .name= "SRC2_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC2_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_alu_src0_src1_src2__default_gen_0 = {
       .display  = "{NAME}{DST_FULL}{SAT}{COND}{SKPHP}{UNK}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, {SRC0}, {SRC1}, {SRC2}",
       .num_fields = 12,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src0_src1_src2__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src0_src1_src2__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC2_REG", .low = 100, .high = 108,
            .type = TYPE_UINT,
          },
          { .name = "SRC2", .low = 110, .high = 119,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_alu_src0_src1_src2__default_gen_0_src2,
          },
          { .name = "SRC2_AMODE", .low = 121, .high = 123,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC2_RGROUP", .low = 124, .high = 126,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_alu_src0_src1_src2_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#instruction-alu-src0-src1-src2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf0000000, 0xf78, 0x40, 0x8 },
       .decode = decode___instruction_alu_src0_src1_src2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_alu_src0_src1_src2_gen_0_fields),
       .decode_fields = decode___instruction_alu_src0_src1_src2_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_alu_src0_src1_src2__default_gen_0,
       },
};
static const struct isa_field_params __instruction_tex__default_gen_0_dst = {
       .num_params = 2,
       .params = {
           { .name= "DST_USE",  .as = "DST_USE" },
           { .name= "COMPS",  .as = "COMPS" },

       },
};
static const struct isa_case __instruction_tex__default_gen_0 = {
       .num_fields = 8,
       .fields   = {
          { .name = "THREAD", .low = 0, .high = 0,
            .expr = &expr_anon_12,
            .type = TYPE_ENUM,
            .enums = &enum___thread,
          },
          { .name = "DST_USE", .low = 12, .high = 12,
            .type = TYPE_BOOL,
          },
          { .name = "DST", .low = 13, .high = 26,
            .type = TYPE_BITSET,
            .bitsets = __instruction_dst,
            .params = &__instruction_tex__default_gen_0_dst,
          },
          { .name = "TEX_ID", .low = 27, .high = 31,
            .type = TYPE_UINT,
          },
          { .name = "RMODE", .low = 32, .high = 34,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "TEX_SWIZ", .low = 35, .high = 42,
            .type = TYPE_BITSET,
            .bitsets = __src_swizzle,
          },
          { .name = "LOW_HALF", .low = 109, .high = 109,
            .type = TYPE_BOOL,
          },
          { .name = "HIGH_HALF", .low = 120, .high = 120,
            .type = TYPE_BOOL,
          },
       },
};
static const struct isa_bitset bitset___instruction_tex_gen_0 = {

       .parent   = &bitset___instruction_gen_0,
       .name     = "#instruction-tex",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x0, 0x0, 0x80000000 },
       .decode = decode___instruction_tex_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_tex_gen_0_fields),
       .decode_fields = decode___instruction_tex_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_tex__default_gen_0,
       },
};
static const struct isa_case __instruction_tex_src0_maybe_src2__case0_gen_0 = {
       .expr     = &expr___instruction_has_src2,
       .display  = "{NAME}{THREAD} {DST:align=18}, tex{TEX_ID}{TEX_SWIZ}, {SRC0}, void, {SRC2}",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_field_params __instruction_tex_src0_maybe_src2__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_tex_src0_maybe_src2__default_gen_0_src2 = {
       .num_params = 3,
       .params = {
           { .name= "SRC2_REG",  .as = "SRC_REG" },
           { .name= "SRC2_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC2_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_tex_src0_maybe_src2__default_gen_0 = {
       .display  = "{NAME}{THREAD} {DST:align=18}, tex{TEX_ID}{TEX_SWIZ}, {SRC0}, void, void",
       .num_fields = 9,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_maybe_src2__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC2_USE", .low = 99, .high = 99,
            .type = TYPE_BOOL,
          },
          { .name = "SRC2_REG", .low = 100, .high = 108,
            .type = TYPE_UINT,
          },
          { .name = "SRC2", .low = 110, .high = 119,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_maybe_src2__default_gen_0_src2,
          },
          { .name = "SRC2_AMODE", .low = 121, .high = 123,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC2_RGROUP", .low = 124, .high = 126,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_tex_src0_maybe_src2_gen_0 = {

       .parent   = &bitset___instruction_tex_gen_0,
       .name     = "#instruction-tex-src0-maybe-src2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x800, 0x3ffeffc0, 0x80000007 },
       .decode = decode___instruction_tex_src0_maybe_src2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_tex_src0_maybe_src2_gen_0_fields),
       .decode_fields = decode___instruction_tex_src0_maybe_src2_gen_0_fields,
       .num_cases = 2,
       .cases    = {
            &__instruction_tex_src0_maybe_src2__case0_gen_0,
            &__instruction_tex_src0_maybe_src2__default_gen_0,
       },
};
static const struct isa_field_params __instruction_tex_src0_src1__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_tex_src0_src1__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_tex_src0_src1__default_gen_0 = {
       .display  = "{NAME}{THREAD} {DST:align=18}, tex{TEX_ID}{TEX_SWIZ}, {SRC0}, {SRC1}, void",
       .num_fields = 8,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_src1__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_src1__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_tex_src0_src1_gen_0 = {

       .parent   = &bitset___instruction_tex_gen_0,
       .name     = "#instruction-tex-src0-src1",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x800, 0x40, 0xfeffdff8 },
       .decode = decode___instruction_tex_src0_src1_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_tex_src0_src1_gen_0_fields),
       .decode_fields = decode___instruction_tex_src0_src1_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_tex_src0_src1__default_gen_0,
       },
};
static const struct isa_field_params __instruction_tex_src0_src1_src2__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_tex_src0_src1_src2__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_tex_src0_src1_src2__default_gen_0_src2 = {
       .num_params = 3,
       .params = {
           { .name= "SRC2_REG",  .as = "SRC_REG" },
           { .name= "SRC2_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC2_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_tex_src0_src1_src2__default_gen_0 = {
       .display  = "{NAME}{THREAD} {DST:align=18}, tex{TEX_ID}{TEX_SWIZ}, {SRC0}, {SRC1}, {SRC2}",
       .num_fields = 12,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_src1_src2__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_src1_src2__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC2_REG", .low = 100, .high = 108,
            .type = TYPE_UINT,
          },
          { .name = "SRC2", .low = 110, .high = 119,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_src1_src2__default_gen_0_src2,
          },
          { .name = "SRC2_AMODE", .low = 121, .high = 123,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC2_RGROUP", .low = 124, .high = 126,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_tex_src0_src1_src2_gen_0 = {

       .parent   = &bitset___instruction_tex_gen_0,
       .name     = "#instruction-tex-src0-src1-src2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x800, 0x40, 0x80000008 },
       .decode = decode___instruction_tex_src0_src1_src2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_tex_src0_src1_src2_gen_0_fields),
       .decode_fields = decode___instruction_tex_src0_src1_src2_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_tex_src0_src1_src2__default_gen_0,
       },
};
static const struct isa_case __instruction_tex_src0_maybe_src1_src2__case0_gen_0 = {
       .expr     = &expr___instruction_has_src0,
       .display  = "{NAME}{THREAD} {DST:align=18}, tex{TEX_ID}{TEX_SWIZ}, {SRC0}, void, void",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_case __instruction_tex_src0_maybe_src1_src2__case1_gen_0 = {
       .expr     = &expr___instruction_has_src0_src1_src2,
       .display  = "{NAME}{THREAD} {DST:align=18}, tex{TEX_ID}{TEX_SWIZ}, {SRC0}, {SRC1}, {SRC2}",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_field_params __instruction_tex_src0_maybe_src1_src2__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_tex_src0_maybe_src1_src2__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_tex_src0_maybe_src1_src2__default_gen_0_src2 = {
       .num_params = 3,
       .params = {
           { .name= "SRC2_REG",  .as = "SRC_REG" },
           { .name= "SRC2_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC2_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_tex_src0_maybe_src1_src2__default_gen_0 = {
       .display  = "{NAME}{THREAD} {DST:align=18}, tex{TEX_ID}{TEX_SWIZ}, {SRC0}, {SRC1}, void",
       .num_fields = 15,
       .fields   = {
          { .name = "SRC0_USE", .low = 43, .high = 43,
            .type = TYPE_BOOL,
          },
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_maybe_src1_src2__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_USE", .low = 70, .high = 70,
            .type = TYPE_BOOL,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_maybe_src1_src2__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC2_USE", .low = 99, .high = 99,
            .type = TYPE_BOOL,
          },
          { .name = "SRC2_REG", .low = 100, .high = 108,
            .type = TYPE_UINT,
          },
          { .name = "SRC2", .low = 110, .high = 119,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_tex_src0_maybe_src1_src2__default_gen_0_src2,
          },
          { .name = "SRC2_AMODE", .low = 121, .high = 123,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC2_RGROUP", .low = 124, .high = 126,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_tex_src0_maybe_src1_src2_gen_0 = {

       .parent   = &bitset___instruction_tex_gen_0,
       .name     = "#instruction-tex-src0-maybe-src1-src2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x0, 0x0, 0x80000000 },
       .decode = decode___instruction_tex_src0_maybe_src1_src2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_tex_src0_maybe_src1_src2_gen_0_fields),
       .decode_fields = decode___instruction_tex_src0_maybe_src1_src2_gen_0_fields,
       .num_cases = 3,
       .cases    = {
            &__instruction_tex_src0_maybe_src1_src2__case0_gen_0,
            &__instruction_tex_src0_maybe_src1_src2__case1_gen_0,
            &__instruction_tex_src0_maybe_src1_src2__default_gen_0,
       },
};
static const struct isa_case __instruction_cf__default_gen_0 = {
       .num_fields = 1,
       .fields   = {
          { .name = "TARGET", .low = 103, .high = 117,
            .type = TYPE_ABSBRANCH,
            .call = false,
          },
       },
};
static const struct isa_bitset bitset___instruction_cf_gen_0 = {

       .parent   = &bitset___instruction_gen_0,
       .name     = "#instruction-cf",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff000, 0x7ff, 0x0, 0xffc00078 },
       .decode = decode___instruction_cf_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_cf_gen_0_fields),
       .decode_fields = decode___instruction_cf_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_cf__default_gen_0,
       },
};
static const struct isa_case __instruction_cf_no_src__default_gen_0 = {
       .display  = "{NAME}{COND}{TYPE} {:align=18}void, void, void, {TARGET}",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset___instruction_cf_no_src_gen_0 = {

       .parent   = &bitset___instruction_cf_gen_0,
       .name     = "#instruction-cf-no-src",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff000, 0xffdfffff, 0x3ffeffff, 0xffc0007f },
       .decode = decode___instruction_cf_no_src_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_cf_no_src_gen_0_fields),
       .decode_fields = decode___instruction_cf_no_src_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_cf_no_src__default_gen_0,
       },
};
static const struct isa_field_params __instruction_cf_src0__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_cf_src0__default_gen_0 = {
       .display  = "{NAME}{COND}{TYPE} {:align=18}void, {SRC0}, void, {TARGET}",
       .num_fields = 4,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_cf_src0__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_cf_src0_gen_0 = {

       .parent   = &bitset___instruction_cf_gen_0,
       .name     = "#instruction-cf-src0",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff000, 0xfff, 0x3ffeffc0, 0xffc0007f },
       .decode = decode___instruction_cf_src0_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_cf_src0_gen_0_fields),
       .decode_fields = decode___instruction_cf_src0_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_cf_src0__default_gen_0,
       },
};
static const struct isa_field_params __instruction_cf_src0_src1__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_cf_src0_src1__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_cf_src0_src1__default_gen_0 = {
       .display  = "{NAME}{COND}{TYPE} {:align=18}void, {SRC0}, {SRC1}, {TARGET}",
       .num_fields = 8,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_cf_src0_src1__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_cf_src0_src1__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_cf_src0_src1_gen_0 = {

       .parent   = &bitset___instruction_cf_gen_0,
       .name     = "#instruction-cf-src0-src1",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff000, 0xfff, 0x40, 0xffc00078 },
       .decode = decode___instruction_cf_src0_src1_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_cf_src0_src1_gen_0_fields),
       .decode_fields = decode___instruction_cf_src0_src1_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_cf_src0_src1__default_gen_0,
       },
};
static const struct isa_case __left_shift__case0_gen_0 = {
       .expr     = &expr_anon_13,
       .display  = ".ls{LEFT_SHIFT}",
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_case __left_shift__default_gen_0 = {
       .display  = "",
       .num_fields = 1,
       .fields   = {
          { .name = "LEFT_SHIFT", .low = 0, .high = 2,
            .type = TYPE_UINT,
          },
       },
};
static const struct isa_bitset bitset___left_shift_gen_0 = {

       .name     = "#left-shift",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x0, 0x0, 0x0, 0x0 },
       .decode = decode___left_shift_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___left_shift_gen_0_fields),
       .decode_fields = decode___left_shift_gen_0_fields,
       .num_cases = 2,
       .cases    = {
            &__left_shift__case0_gen_0,
            &__left_shift__default_gen_0,
       },
};
static const struct isa_field_params __instruction_load__default_gen_0_dst = {
       .num_params = 2,
       .params = {
           { .name= "DST_USE",  .as = "DST_USE" },
           { .name= "COMPS",  .as = "COMPS" },

       },
};
static const struct isa_field_params __instruction_load__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_load__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_load__default_gen_0 = {
       .display  = "{NAME}{SAT}{COND}{SKPHP}{DENORM}{LOCAL}{TYPE}{LEFT_SHIFT}{PMODE} {DST:align=18}, {SRC0}, {SRC1}, void",
       .num_fields = 19,
       .fields   = {
          { .name = "THREAD", .low = 0, .high = 0,
            .expr = &expr_anon_14,
            .type = TYPE_ENUM,
            .enums = &enum___thread,
          },
          { .name = "DST_USE", .low = 12, .high = 12,
            .type = TYPE_BOOL,
          },
          { .name = "DST", .low = 13, .high = 26,
            .type = TYPE_BITSET,
            .bitsets = __instruction_dst,
            .params = &__instruction_load__default_gen_0_dst,
          },
          { .name = "PMODE", .low = 34, .high = 34,
            .display = ".pack",
            .type = TYPE_BOOL_INV,
          },
          { .name = "LEFT_SHIFT", .low = 35, .high = 37,
            .type = TYPE_BITSET,
            .bitsets = __left_shift,
          },
          { .name = "SKPHP", .low = 39, .high = 39,
            .display = ".skpHp",
            .type = TYPE_BOOL,
          },
          { .name = "LOCAL", .low = 40, .high = 40,
            .display = ".local",
            .type = TYPE_BOOL,
          },
          { .name = "DENORM", .low = 42, .high = 42,
            .display = ".denorm",
            .type = TYPE_BOOL,
          },
          { .name = "LOW_HALF", .low = 109, .high = 109,
            .type = TYPE_BOOL,
          },
          { .name = "HIGH_HALF", .low = 120, .high = 120,
            .type = TYPE_BOOL,
          },
          { .name = "DST_FULL", .low = 127, .high = 127,
            .display = ".hp",
            .type = TYPE_BOOL,
          },
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_load__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_load__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_load_gen_0 = {

       .parent   = &bitset___instruction_gen_0,
       .name     = "#instruction-load",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf8000000, 0xa43, 0x40, 0x7effdff8 },
       .decode = decode___instruction_load_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_load_gen_0_fields),
       .decode_fields = decode___instruction_load_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_load__default_gen_0,
       },
};
static const struct isa_field_params __instruction_store__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_store__default_gen_0_src1 = {
       .num_params = 3,
       .params = {
           { .name= "SRC1_REG",  .as = "SRC_REG" },
           { .name= "SRC1_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC1_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_field_params __instruction_store__default_gen_0_src2 = {
       .num_params = 3,
       .params = {
           { .name= "SRC2_REG",  .as = "SRC_REG" },
           { .name= "SRC2_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC2_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __instruction_store__default_gen_0 = {
       .display  = "{NAME}{SAT}{COND}{SKPHP}{DENORM}{LOCAL}{TYPE}{LEFT_SHIFT}{PMODE} {:align=18}mem{COMPS}, {SRC0}, {SRC1}, {SRC2}",
       .num_fields = 22,
       .fields   = {
          { .name = "THREAD", .low = 0, .high = 0,
            .expr = &expr_anon_15,
            .type = TYPE_ENUM,
            .enums = &enum___thread,
          },
          { .name = "COMPS", .low = 23, .high = 26,
            .type = TYPE_ENUM,
            .enums = &enum___wrmask,
          },
          { .name = "PMODE", .low = 34, .high = 34,
            .display = ".pack",
            .type = TYPE_BOOL_INV,
          },
          { .name = "LEFT_SHIFT", .low = 35, .high = 37,
            .type = TYPE_BITSET,
            .bitsets = __left_shift,
          },
          { .name = "SKPHP", .low = 39, .high = 39,
            .display = ".skpHp",
            .type = TYPE_BOOL,
          },
          { .name = "LOCAL", .low = 40, .high = 40,
            .display = ".local",
            .type = TYPE_BOOL,
          },
          { .name = "DENORM", .low = 42, .high = 42,
            .display = ".denorm",
            .type = TYPE_BOOL,
          },
          { .name = "LOW_HALF", .low = 109, .high = 109,
            .type = TYPE_BOOL,
          },
          { .name = "HIGH_HALF", .low = 120, .high = 120,
            .type = TYPE_BOOL,
          },
          { .name = "DST_FULL", .low = 127, .high = 127,
            .display = ".hp",
            .type = TYPE_BOOL,
          },
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_store__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC1_REG", .low = 71, .high = 79,
            .type = TYPE_UINT,
          },
          { .name = "SRC1", .low = 81, .high = 90,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_store__default_gen_0_src1,
          },
          { .name = "SRC1_AMODE", .low = 91, .high = 93,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC1_RGROUP", .low = 96, .high = 98,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
          { .name = "SRC2_REG", .low = 100, .high = 108,
            .type = TYPE_UINT,
          },
          { .name = "SRC2", .low = 110, .high = 119,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__instruction_store__default_gen_0_src2,
          },
          { .name = "SRC2_AMODE", .low = 121, .high = 123,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC2_RGROUP", .low = 124, .high = 126,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___instruction_store_gen_0 = {

       .parent   = &bitset___instruction_gen_0,
       .name     = "#instruction-store",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0xf87ff000, 0x203, 0x0, 0x0 },
       .mask.bitset     = { 0xf87ff000, 0xa43, 0x40, 0x8 },
       .decode = decode___instruction_store_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___instruction_store_gen_0_fields),
       .decode_fields = decode___instruction_store_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__instruction_store__default_gen_0,
       },
};
static const struct isa_case nop__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_nop_gen_0 = {

       .parent   = &bitset___instruction_alu_no_src_gen_0,
       .name     = "nop",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0x7effdfff },
       .decode = decode_nop_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_nop_gen_0_fields),
       .decode_fields = decode_nop_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &nop__default_gen_0,
       },
};
static const struct isa_case add__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_add_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "add",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1, 0x800, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_add_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_add_gen_0_fields),
       .decode_fields = decode_add_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &add__default_gen_0,
       },
};
static const struct isa_case mad__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_mad_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "mad",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_mad_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_mad_gen_0_fields),
       .decode_fields = decode_mad_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &mad__default_gen_0,
       },
};
static const struct isa_case mul__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_mul_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "mul",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_mul_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_mul_gen_0_fields),
       .decode_fields = decode_mul_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &mul__default_gen_0,
       },
};
static const struct isa_case dp3__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_dp3_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "dp3",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x5, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_dp3_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_dp3_gen_0_fields),
       .decode_fields = decode_dp3_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &dp3__default_gen_0,
       },
};
static const struct isa_case dp4__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_dp4_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "dp4",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x6, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_dp4_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_dp4_gen_0_fields),
       .decode_fields = decode_dp4_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &dp4__default_gen_0,
       },
};
static const struct isa_case dsx__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_dsx_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "dsx",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x7, 0x800, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_dsx_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_dsx_gen_0_fields),
       .decode_fields = decode_dsx_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &dsx__default_gen_0,
       },
};
static const struct isa_case dsy__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_dsy_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "dsy",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x8, 0x800, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_dsy_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_dsy_gen_0_fields),
       .decode_fields = decode_dsy_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &dsy__default_gen_0,
       },
};
static const struct isa_case mov__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_mov_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "mov",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x9, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_mov_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_mov_gen_0_fields),
       .decode_fields = decode_mov_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &mov__default_gen_0,
       },
};
static const struct isa_case movar__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_movar_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "movar",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0xa, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_movar_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_movar_gen_0_fields),
       .decode_fields = decode_movar_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &movar__default_gen_0,
       },
};
static const struct isa_case rcp__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_rcp_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "rcp",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0xc, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_rcp_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_rcp_gen_0_fields),
       .decode_fields = decode_rcp_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &rcp__default_gen_0,
       },
};
static const struct isa_case rsq__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_rsq_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "rsq",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0xd, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_rsq_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_rsq_gen_0_fields),
       .decode_fields = decode_rsq_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &rsq__default_gen_0,
       },
};
static const struct isa_case select__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_select_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "select",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0xf, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_select_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_select_gen_0_fields),
       .decode_fields = decode_select_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &select__default_gen_0,
       },
};
static const struct isa_case set__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_set_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "set",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x10, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_set_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_set_gen_0_fields),
       .decode_fields = decode_set_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &set__default_gen_0,
       },
};
static const struct isa_case exp__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_exp_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "exp",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x11, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_exp_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_exp_gen_0_fields),
       .decode_fields = decode_exp_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &exp__default_gen_0,
       },
};
static const struct isa_case log__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_log_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "log",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x12, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_log_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_log_gen_0_fields),
       .decode_fields = decode_log_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &log__default_gen_0,
       },
};
static const struct isa_case frc__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_frc_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "frc",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x13, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_frc_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_frc_gen_0_fields),
       .decode_fields = decode_frc_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &frc__default_gen_0,
       },
};
static const struct isa_case call__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_call_gen_0 = {

       .parent   = &bitset___instruction_cf_no_src_gen_0,
       .name     = "call",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x14, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff03f, 0xffdfffff, 0x3fffffff, 0xffc0007f },
       .decode = decode_call_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_call_gen_0_fields),
       .decode_fields = decode_call_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &call__default_gen_0,
       },
};
static const struct isa_case ret__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_ret_gen_0 = {

       .parent   = &bitset___instruction_alu_no_src_gen_0,
       .name     = "ret",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x15, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0x7effdfff },
       .decode = decode_ret_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_ret_gen_0_fields),
       .decode_fields = decode_ret_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &ret__default_gen_0,
       },
};
static const struct isa_case branch__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_branch_gen_0 = {

       .parent   = &bitset___instruction_cf_no_src_gen_0,
       .name     = "branch",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x16, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff03f, 0xffdfffff, 0x3fffffff, 0xffc0007f },
       .decode = decode_branch_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_branch_gen_0_fields),
       .decode_fields = decode_branch_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &branch__default_gen_0,
       },
};
static const struct isa_case branch_unary__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_branch_unary_gen_0 = {

       .parent   = &bitset___instruction_cf_src0_gen_0,
       .name     = "branch",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x16, 0x800, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff03f, 0xfff, 0x3fffffc0, 0xffc0007f },
       .decode = decode_branch_unary_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_branch_unary_gen_0_fields),
       .decode_fields = decode_branch_unary_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &branch_unary__default_gen_0,
       },
};
static const struct isa_case branch_binary__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_branch_binary_gen_0 = {

       .parent   = &bitset___instruction_cf_src0_src1_gen_0,
       .name     = "branch",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x16, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff03f, 0xfff, 0x10040, 0xffc00078 },
       .decode = decode_branch_binary_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_branch_binary_gen_0_fields),
       .decode_fields = decode_branch_binary_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &branch_binary__default_gen_0,
       },
};
static const struct isa_case texkill__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_texkill_gen_0 = {

       .parent   = &bitset___instruction_alu_no_dst_maybe_src0_src1_gen_0,
       .name     = "texkill",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x17, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0x778, 0x10000, 0x7effdff8 },
       .decode = decode_texkill_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_texkill_gen_0_fields),
       .decode_fields = decode_texkill_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &texkill__default_gen_0,
       },
};
static const struct isa_case texld__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_texld_gen_0 = {

       .parent   = &bitset___instruction_tex_src0_maybe_src2_gen_0,
       .name     = "texld",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x18, 0x800, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x3f, 0x800, 0x3fffffc0, 0x80000007 },
       .decode = decode_texld_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_texld_gen_0_fields),
       .decode_fields = decode_texld_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &texld__default_gen_0,
       },
};
static const struct isa_case texldb__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_texldb_gen_0 = {

       .parent   = &bitset___instruction_tex_src0_maybe_src1_src2_gen_0,
       .name     = "texldb",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x19, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x3f, 0x0, 0x10000, 0x80000000 },
       .decode = decode_texldb_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_texldb_gen_0_fields),
       .decode_fields = decode_texldb_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &texldb__default_gen_0,
       },
};
static const struct isa_case texldd__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_texldd_gen_0 = {

       .parent   = &bitset___instruction_tex_src0_src1_src2_gen_0,
       .name     = "texldd",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1a, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x3f, 0x800, 0x10040, 0x80000008 },
       .decode = decode_texldd_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_texldd_gen_0_fields),
       .decode_fields = decode_texldd_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &texldd__default_gen_0,
       },
};
static const struct isa_case texldl__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_texldl_gen_0 = {

       .parent   = &bitset___instruction_tex_src0_maybe_src1_src2_gen_0,
       .name     = "texldl",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1b, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x3f, 0x0, 0x10000, 0x80000000 },
       .decode = decode_texldl_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_texldl_gen_0_fields),
       .decode_fields = decode_texldl_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &texldl__default_gen_0,
       },
};
static const struct isa_case sqrt__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_sqrt_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "sqrt",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x21, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_sqrt_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_sqrt_gen_0_fields),
       .decode_fields = decode_sqrt_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &sqrt__default_gen_0,
       },
};
static const struct isa_case sin__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_sin_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "sin",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x22, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_sin_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_sin_gen_0_fields),
       .decode_fields = decode_sin_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &sin__default_gen_0,
       },
};
static const struct isa_case cos__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_cos_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "cos",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x23, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_cos_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_cos_gen_0_fields),
       .decode_fields = decode_cos_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &cos__default_gen_0,
       },
};
static const struct isa_case branch_any__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_branch_any_gen_0 = {

       .parent   = &bitset___instruction_cf_src0_src1_gen_0,
       .name     = "branch_any",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x24, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x4, 0x0, 0x0 },
       .mask.bitset     = { 0xfffff03f, 0xfff, 0x10040, 0xffc00078 },
       .decode = decode_branch_any_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_branch_any_gen_0_fields),
       .decode_fields = decode_branch_any_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &branch_any__default_gen_0,
       },
};
static const struct isa_case floor__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_floor_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "floor",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x25, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_floor_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_floor_gen_0_fields),
       .decode_fields = decode_floor_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &floor__default_gen_0,
       },
};
static const struct isa_case ceil__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_ceil_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "ceil",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x26, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_ceil_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_ceil_gen_0_fields),
       .decode_fields = decode_ceil_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &ceil__default_gen_0,
       },
};
static const struct isa_case sign__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_sign_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "sign",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x27, 0x0, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_sign_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_sign_gen_0_fields),
       .decode_fields = decode_sign_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &sign__default_gen_0,
       },
};
static const struct isa_case barrier__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_barrier_gen_0 = {

       .parent   = &bitset___instruction_alu_no_src_gen_0,
       .name     = "barrier",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2a, 0x0, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0x7effdfff },
       .decode = decode_barrier_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_barrier_gen_0_fields),
       .decode_fields = decode_barrier_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &barrier__default_gen_0,
       },
};
static const struct isa_case swizzle__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_swizzle_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "swizzle",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2b, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_swizzle_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_swizzle_gen_0_fields),
       .decode_fields = decode_swizzle_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &swizzle__default_gen_0,
       },
};
static const struct isa_case i2i__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_i2i_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "i2i",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2c, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_i2i_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_i2i_gen_0_fields),
       .decode_fields = decode_i2i_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &i2i__default_gen_0,
       },
};
static const struct isa_case i2f__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_i2f_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_gen_0,
       .name     = "i2f",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2d, 0x800, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_i2f_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_i2f_gen_0_fields),
       .decode_fields = decode_i2f_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &i2f__default_gen_0,
       },
};
static const struct isa_case f2i__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_f2i_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_gen_0,
       .name     = "f2i",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2e, 0x800, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_f2i_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_f2i_gen_0_fields),
       .decode_fields = decode_f2i_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &f2i__default_gen_0,
       },
};
static const struct isa_case f2irnd__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_f2irnd_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_gen_0,
       .name     = "f2irnd",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2f, 0x800, 0x0, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_f2irnd_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_f2irnd_gen_0_fields),
       .decode_fields = decode_f2irnd_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &f2irnd__default_gen_0,
       },
};
static const struct isa_case cmp__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_cmp_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "cmp",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x31, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_cmp_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_cmp_gen_0_fields),
       .decode_fields = decode_cmp_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &cmp__default_gen_0,
       },
};
static const struct isa_case load__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_load_gen_0 = {

       .parent   = &bitset___instruction_load_gen_0,
       .name     = "load",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x32, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf800003f, 0xa43, 0x10040, 0x7effdff8 },
       .decode = decode_load_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_load_gen_0_fields),
       .decode_fields = decode_load_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &load__default_gen_0,
       },
};
static const struct isa_case store__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_store_gen_0 = {

       .parent   = &bitset___instruction_store_gen_0,
       .name     = "store",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x33, 0x800, 0x40, 0x8 },
       .dontcare.bitset = { 0xf87ff000, 0x203, 0x0, 0x0 },
       .mask.bitset     = { 0xf87ff03f, 0xa43, 0x10040, 0x8 },
       .decode = decode_store_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_store_gen_0_fields),
       .decode_fields = decode_store_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &store__default_gen_0,
       },
};
static const struct isa_case clamp0_max__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_clamp0_max_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "clamp0_max",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x36, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_clamp0_max_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_clamp0_max_gen_0_fields),
       .decode_fields = decode_clamp0_max_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &clamp0_max__default_gen_0,
       },
};
static const struct isa_case iaddsat__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_iaddsat_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "iaddsat",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3b, 0x800, 0x0, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_iaddsat_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_iaddsat_gen_0_fields),
       .decode_fields = decode_iaddsat_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &iaddsat__default_gen_0,
       },
};
static const struct isa_case imullo0__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_imullo0_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "imullo0",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3c, 0x800, 0x40, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_imullo0_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_imullo0_gen_0_fields),
       .decode_fields = decode_imullo0_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &imullo0__default_gen_0,
       },
};
static const struct isa_case imulhi0__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_imulhi0_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "imulhi0",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x0, 0x800, 0x10040, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_imulhi0_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_imulhi0_gen_0_fields),
       .decode_fields = decode_imulhi0_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &imulhi0__default_gen_0,
       },
};
static const struct isa_case idiv0__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_idiv0_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "idiv0",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x4, 0x800, 0x10040, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_idiv0_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_idiv0_gen_0_fields),
       .decode_fields = decode_idiv0_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &idiv0__default_gen_0,
       },
};
static const struct isa_case imod__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_imod_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "imod",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x8, 0x800, 0x10040, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_imod_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_imod_gen_0_fields),
       .decode_fields = decode_imod_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &imod__default_gen_0,
       },
};
static const struct isa_case txf__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_txf_gen_0 = {

       .parent   = &bitset___instruction_tex_src0_src1_src2_gen_0,
       .name     = "txf",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x9, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x3f, 0x800, 0x10040, 0x80000008 },
       .decode = decode_txf_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_txf_gen_0_fields),
       .decode_fields = decode_txf_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &txf__default_gen_0,
       },
};
static const struct isa_case imadlo0__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_imadlo0_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "imadlo0",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0xc, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_imadlo0_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_imadlo0_gen_0_fields),
       .decode_fields = decode_imadlo0_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &imadlo0__default_gen_0,
       },
};
static const struct isa_case imadlosat0__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_imadlosat0_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "imadlosat0",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0xe, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_imadlosat0_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_imadlosat0_gen_0_fields),
       .decode_fields = decode_imadlosat0_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &imadlosat0__default_gen_0,
       },
};
static const struct isa_case movai__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_movai_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "movai",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x16, 0x0, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_movai_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_movai_gen_0_fields),
       .decode_fields = decode_movai_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &movai__default_gen_0,
       },
};
static const struct isa_case iabs__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_iabs_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "iabs",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x17, 0x0, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_iabs_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_iabs_gen_0_fields),
       .decode_fields = decode_iabs_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &iabs__default_gen_0,
       },
};
static const struct isa_case leadzero__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_leadzero_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "leadzero",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x18, 0x0, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_leadzero_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_leadzero_gen_0_fields),
       .decode_fields = decode_leadzero_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &leadzero__default_gen_0,
       },
};
static const struct isa_case lshift__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_lshift_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "lshift",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x19, 0x800, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_lshift_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_lshift_gen_0_fields),
       .decode_fields = decode_lshift_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &lshift__default_gen_0,
       },
};
static const struct isa_case rshift__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_rshift_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "rshift",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1a, 0x800, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_rshift_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_rshift_gen_0_fields),
       .decode_fields = decode_rshift_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &rshift__default_gen_0,
       },
};
static const struct isa_case rotate__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_rotate_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "rotate",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1b, 0x800, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_rotate_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_rotate_gen_0_fields),
       .decode_fields = decode_rotate_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &rotate__default_gen_0,
       },
};
static const struct isa_case or__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_or_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "or",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1c, 0x800, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_or_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_or_gen_0_fields),
       .decode_fields = decode_or_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &or__default_gen_0,
       },
};
static const struct isa_case and__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_and_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "and",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1d, 0x800, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_and_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_and_gen_0_fields),
       .decode_fields = decode_and_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &and__default_gen_0,
       },
};
static const struct isa_case xor__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_xor_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src2_gen_0,
       .name     = "xor",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1e, 0x800, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0xf },
       .decode = decode_xor_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_xor_gen_0_fields),
       .decode_fields = decode_xor_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &xor__default_gen_0,
       },
};
static const struct isa_case not__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_not_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "not",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x1f, 0x0, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_not_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_not_gen_0_fields),
       .decode_fields = decode_not_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &not__default_gen_0,
       },
};
static const struct isa_case bit_extract__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_bit_extract_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "bit_extract",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x20, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_bit_extract_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_bit_extract_gen_0_fields),
       .decode_fields = decode_bit_extract_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &bit_extract__default_gen_0,
       },
};
static const struct isa_case popcount__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_popcount_gen_0 = {

       .parent   = &bitset___instruction_alu_src2_gen_0,
       .name     = "popcount",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x21, 0x0, 0x10000, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x3fffffff, 0xf },
       .decode = decode_popcount_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_popcount_gen_0_fields),
       .decode_fields = decode_popcount_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &popcount__default_gen_0,
       },
};
static const struct isa_case div__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_div_gen_0 = {

       .parent   = &bitset___instruction_alu_src1_src2_gen_0,
       .name     = "div",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x24, 0x0, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xffdfff78, 0x1007f, 0x8 },
       .decode = decode_div_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_div_gen_0_fields),
       .decode_fields = decode_div_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &div__default_gen_0,
       },
};
static const struct isa_case atomic_add__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_atomic_add_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "atomic_add",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x25, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_atomic_add_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_atomic_add_gen_0_fields),
       .decode_fields = decode_atomic_add_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &atomic_add__default_gen_0,
       },
};
static const struct isa_case atomic_xchg__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_atomic_xchg_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "atomic_xchg",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x26, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_atomic_xchg_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_atomic_xchg_gen_0_fields),
       .decode_fields = decode_atomic_xchg_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &atomic_xchg__default_gen_0,
       },
};
static const struct isa_case atomic_cmp_xchg__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_atomic_cmp_xchg_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "atomic_cmp_xchg",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x27, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_atomic_cmp_xchg_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_atomic_cmp_xchg_gen_0_fields),
       .decode_fields = decode_atomic_cmp_xchg_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &atomic_cmp_xchg__default_gen_0,
       },
};
static const struct isa_case atomic_min__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_atomic_min_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "atomic_min",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x28, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_atomic_min_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_atomic_min_gen_0_fields),
       .decode_fields = decode_atomic_min_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &atomic_min__default_gen_0,
       },
};
static const struct isa_case atomic_max__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_atomic_max_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "atomic_max",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x29, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_atomic_max_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_atomic_max_gen_0_fields),
       .decode_fields = decode_atomic_max_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &atomic_max__default_gen_0,
       },
};
static const struct isa_case atomic_or__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_atomic_or_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "atomic_or",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2a, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_atomic_or_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_atomic_or_gen_0_fields),
       .decode_fields = decode_atomic_or_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &atomic_or__default_gen_0,
       },
};
static const struct isa_case atomic_and__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_atomic_and_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "atomic_and",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2b, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_atomic_and_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_atomic_and_gen_0_fields),
       .decode_fields = decode_atomic_and_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &atomic_and__default_gen_0,
       },
};
static const struct isa_case atomic_xor__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_atomic_xor_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_src2_gen_0,
       .name     = "atomic_xor",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2c, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x8 },
       .decode = decode_atomic_xor_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_atomic_xor_gen_0_fields),
       .decode_fields = decode_atomic_xor_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &atomic_xor__default_gen_0,
       },
};
static const struct isa_case bit_rev__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_bit_rev_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_gen_0,
       .name     = "bit_rev",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2d, 0x800, 0x10000, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_bit_rev_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_bit_rev_gen_0_fields),
       .decode_fields = decode_bit_rev_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &bit_rev__default_gen_0,
       },
};
static const struct isa_case texldlpcf__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_texldlpcf_gen_0 = {

       .parent   = &bitset___instruction_tex_src0_src1_src2_gen_0,
       .name     = "texldlpcf",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x2f, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x3f, 0x800, 0x10040, 0x80000008 },
       .decode = decode_texldlpcf_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_texldlpcf_gen_0_fields),
       .decode_fields = decode_texldlpcf_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &texldlpcf__default_gen_0,
       },
};
static const struct isa_case texldgpcf__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_texldgpcf_gen_0 = {

       .parent   = &bitset___instruction_tex_src0_src1_src2_gen_0,
       .name     = "texldgpcf",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x30, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x3f, 0x800, 0x10040, 0x80000008 },
       .decode = decode_texldgpcf_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_texldgpcf_gen_0_fields),
       .decode_fields = decode_texldgpcf_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &texldgpcf__default_gen_0,
       },
};
static const struct isa_case conv__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_conv_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "conv",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x32, 0x800, 0x10040, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_conv_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_conv_gen_0_fields),
       .decode_fields = decode_conv_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &conv__default_gen_0,
       },
};
static const struct isa_case dp2__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_dp2_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "dp2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x33, 0x800, 0x10040, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_dp2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_dp2_gen_0_fields),
       .decode_fields = decode_dp2_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &dp2__default_gen_0,
       },
};
static const struct isa_case norm_dp2__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_norm_dp2_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_gen_0,
       .name     = "norm_dp2",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x34, 0x800, 0x10000, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_norm_dp2_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_norm_dp2_gen_0_fields),
       .decode_fields = decode_norm_dp2_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &norm_dp2__default_gen_0,
       },
};
static const struct isa_case norm_dp3__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_norm_dp3_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_gen_0,
       .name     = "norm_dp3",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x35, 0x800, 0x10000, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_norm_dp3_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_norm_dp3_gen_0_fields),
       .decode_fields = decode_norm_dp3_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &norm_dp3__default_gen_0,
       },
};
static const struct isa_case norm_dp4__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_norm_dp4_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_gen_0,
       .name     = "norm_dp4",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x36, 0x800, 0x10000, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_norm_dp4_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_norm_dp4_gen_0_fields),
       .decode_fields = decode_norm_dp4_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &norm_dp4__default_gen_0,
       },
};
static const struct isa_case norm_mul__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_norm_mul_gen_0 = {

       .parent   = &bitset___instruction_alu_src0_src1_gen_0,
       .name     = "norm_mul",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x37, 0x800, 0x10040, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x10040, 0x7effdff8 },
       .decode = decode_norm_mul_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_norm_mul_gen_0_fields),
       .decode_fields = decode_norm_mul_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &norm_mul__default_gen_0,
       },
};
static const struct isa_case img_load__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_img_load_gen_0 = {

       .parent   = &bitset___instruction_load_gen_0,
       .name     = "img_load",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x39, 0x800, 0x10040, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf800003f, 0xa43, 0x10040, 0x7effdff8 },
       .decode = decode_img_load_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_img_load_gen_0_fields),
       .decode_fields = decode_img_load_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &img_load__default_gen_0,
       },
};
static const struct isa_case img_store__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_img_store_gen_0 = {

       .parent   = &bitset___instruction_store_gen_0,
       .name     = "img_store",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3a, 0x800, 0x10040, 0x8 },
       .dontcare.bitset = { 0xf87ff000, 0x203, 0x0, 0x0 },
       .mask.bitset     = { 0xf87ff03f, 0xa43, 0x10040, 0x8 },
       .decode = decode_img_store_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_img_store_gen_0_fields),
       .decode_fields = decode_img_store_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &img_store__default_gen_0,
       },
};
static const struct isa_case tg4__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_tg4_gen_0 = {

       .parent   = &bitset___instruction_tex_src0_src1_gen_0,
       .name     = "tg4",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3d, 0x800, 0x10040, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0x3f, 0x800, 0x10040, 0xfeffdff8 },
       .decode = decode_tg4_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_tg4_gen_0_fields),
       .decode_fields = decode_tg4_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &tg4__default_gen_0,
       },
};
static const struct isa_case __extended_instruction_alu__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset___extended_instruction_alu_gen_0 = {

       .parent   = &bitset___instruction_alu_gen_0,
       .name     = "#extended-instruction-alu",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3f, 0x0, 0x10000, 0x0 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0x778, 0x10000, 0x0 },
       .decode = decode___extended_instruction_alu_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___extended_instruction_alu_gen_0_fields),
       .decode_fields = decode___extended_instruction_alu_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__extended_instruction_alu__default_gen_0,
       },
};
static const struct isa_field_params __extended_instruction_alu_src0__default_gen_0_src0 = {
       .num_params = 3,
       .params = {
           { .name= "SRC0_REG",  .as = "SRC_REG" },
           { .name= "SRC0_AMODE",  .as = "SRC_AMODE" },
           { .name= "SRC0_RGROUP",  .as = "SRC_RGROUP" },

       },
};
static const struct isa_case __extended_instruction_alu_src0__default_gen_0 = {
       .display  = "{NAME}{DST_FULL}{SAT}{COND}{SKPHP}{UNK}{TYPE}{PMODE}{THREAD}{RMODE} {DST:align=18}, {SRC0}, void, void",
       .num_fields = 4,
       .fields   = {
          { .name = "SRC0_REG", .low = 44, .high = 52,
            .type = TYPE_UINT,
          },
          { .name = "SRC0", .low = 54, .high = 63,
            .type = TYPE_BITSET,
            .bitsets = __instruction_src,
            .params = &__extended_instruction_alu_src0__default_gen_0_src0,
          },
          { .name = "SRC0_AMODE", .low = 64, .high = 66,
            .type = TYPE_ENUM,
            .enums = &enum___reg_addressing_mode,
          },
          { .name = "SRC0_RGROUP", .low = 67, .high = 69,
            .type = TYPE_ENUM,
            .enums = &enum___reg_group,
          },
       },
};
static const struct isa_bitset bitset___extended_instruction_alu_src0_gen_0 = {

       .parent   = &bitset___extended_instruction_alu_gen_0,
       .name     = "#extended-instruction-alu-src0",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3f, 0x800, 0x10000, 0x78000008 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effc00f },
       .decode = decode___extended_instruction_alu_src0_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode___extended_instruction_alu_src0_gen_0_fields),
       .decode_fields = decode___extended_instruction_alu_src0_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &__extended_instruction_alu_src0__default_gen_0,
       },
};
static const struct isa_case bit_findlsb__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_bit_findlsb_gen_0 = {

       .parent   = &bitset___extended_instruction_alu_src0_gen_0,
       .name     = "bit_findlsb",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3f, 0x800, 0x10000, 0x780000b8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_bit_findlsb_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_bit_findlsb_gen_0_fields),
       .decode_fields = decode_bit_findlsb_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &bit_findlsb__default_gen_0,
       },
};
static const struct isa_case bit_findmsb__default_gen_0 = {
       .num_fields = 0,
       .fields   = {
       },
};
static const struct isa_bitset bitset_bit_findmsb_gen_0 = {

       .parent   = &bitset___extended_instruction_alu_src0_gen_0,
       .name     = "bit_findmsb",
       .gen      = {
           .min  = 0,
           .max  = 4294967295,
       },
       .match.bitset    = { 0x3f, 0x800, 0x10000, 0x780000c8 },
       .dontcare.bitset = { 0x0, 0x0, 0x0, 0x0 },
       .mask.bitset     = { 0xf000003f, 0xf78, 0x3fffffc0, 0x7effdfff },
       .decode = decode_bit_findmsb_gen_0,
       .num_decode_fields = ARRAY_SIZE(decode_bit_findmsb_gen_0_fields),
       .decode_fields = decode_bit_findmsb_gen_0_fields,
       .num_cases = 1,
       .cases    = {
            &bit_findmsb__default_gen_0,
       },
};

/*
 * bitset hierarchy root tables (where decoding starts from):
 */

static const struct isa_bitset *__instruction_dst[] = {
             &bitset___instruction_dst_gen_0,
    (void *)0
};
static const struct isa_bitset *__instruction[] = {
             &bitset_nop_gen_0,
             &bitset_add_gen_0,
             &bitset_mad_gen_0,
             &bitset_mul_gen_0,
             &bitset_dp3_gen_0,
             &bitset_dp4_gen_0,
             &bitset_dsx_gen_0,
             &bitset_dsy_gen_0,
             &bitset_mov_gen_0,
             &bitset_movar_gen_0,
             &bitset_rcp_gen_0,
             &bitset_rsq_gen_0,
             &bitset_select_gen_0,
             &bitset_set_gen_0,
             &bitset_exp_gen_0,
             &bitset_log_gen_0,
             &bitset_frc_gen_0,
             &bitset_call_gen_0,
             &bitset_ret_gen_0,
             &bitset_branch_gen_0,
             &bitset_branch_unary_gen_0,
             &bitset_branch_binary_gen_0,
             &bitset_texkill_gen_0,
             &bitset_texld_gen_0,
             &bitset_texldb_gen_0,
             &bitset_texldd_gen_0,
             &bitset_texldl_gen_0,
             &bitset_sqrt_gen_0,
             &bitset_sin_gen_0,
             &bitset_cos_gen_0,
             &bitset_branch_any_gen_0,
             &bitset_floor_gen_0,
             &bitset_ceil_gen_0,
             &bitset_sign_gen_0,
             &bitset_barrier_gen_0,
             &bitset_swizzle_gen_0,
             &bitset_i2i_gen_0,
             &bitset_i2f_gen_0,
             &bitset_f2i_gen_0,
             &bitset_f2irnd_gen_0,
             &bitset_cmp_gen_0,
             &bitset_load_gen_0,
             &bitset_store_gen_0,
             &bitset_clamp0_max_gen_0,
             &bitset_iaddsat_gen_0,
             &bitset_imullo0_gen_0,
             &bitset_imulhi0_gen_0,
             &bitset_idiv0_gen_0,
             &bitset_imod_gen_0,
             &bitset_txf_gen_0,
             &bitset_imadlo0_gen_0,
             &bitset_imadlosat0_gen_0,
             &bitset_movai_gen_0,
             &bitset_iabs_gen_0,
             &bitset_leadzero_gen_0,
             &bitset_lshift_gen_0,
             &bitset_rshift_gen_0,
             &bitset_rotate_gen_0,
             &bitset_or_gen_0,
             &bitset_and_gen_0,
             &bitset_xor_gen_0,
             &bitset_not_gen_0,
             &bitset_bit_extract_gen_0,
             &bitset_popcount_gen_0,
             &bitset_div_gen_0,
             &bitset_atomic_add_gen_0,
             &bitset_atomic_xchg_gen_0,
             &bitset_atomic_cmp_xchg_gen_0,
             &bitset_atomic_min_gen_0,
             &bitset_atomic_max_gen_0,
             &bitset_atomic_or_gen_0,
             &bitset_atomic_and_gen_0,
             &bitset_atomic_xor_gen_0,
             &bitset_bit_rev_gen_0,
             &bitset_texldlpcf_gen_0,
             &bitset_texldgpcf_gen_0,
             &bitset_conv_gen_0,
             &bitset_dp2_gen_0,
             &bitset_norm_dp2_gen_0,
             &bitset_norm_dp3_gen_0,
             &bitset_norm_dp4_gen_0,
             &bitset_norm_mul_gen_0,
             &bitset_img_load_gen_0,
             &bitset_img_store_gen_0,
             &bitset_tg4_gen_0,
             &bitset_bit_findlsb_gen_0,
             &bitset_bit_findmsb_gen_0,
    (void *)0
};
static const struct isa_bitset *__src_swizzle[] = {
             &bitset___src_swizzle_gen_0,
    (void *)0
};
static const struct isa_bitset *__instruction_src[] = {
             &bitset___instruction_src_gen_0,
    (void *)0
};
static const struct isa_bitset *__left_shift[] = {
             &bitset___left_shift_gen_0,
    (void *)0
};

#include "isaspec_decode_impl.c"

static void decode___instruction_dst_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___src_swizzle_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_src_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_no_src_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_no_dst_maybe_src0_src1_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_src0_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_src2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_src0_src1_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_src0_src2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_src1_src2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_alu_src0_src1_src2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_tex_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_tex_src0_maybe_src2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_tex_src0_src1_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_tex_src0_src1_src2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_tex_src0_maybe_src1_src2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_cf_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_cf_no_src_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_cf_src0_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_cf_src0_src1_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___left_shift_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_load_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___instruction_store_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_nop_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_add_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_mad_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_mul_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_dp3_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_dp4_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_dsx_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_dsy_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_mov_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_movar_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_rcp_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_rsq_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_select_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_set_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_exp_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_log_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_frc_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_call_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_ret_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_branch_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_branch_unary_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_branch_binary_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_texkill_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_texld_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_texldb_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_texldd_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_texldl_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_sqrt_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_sin_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_cos_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_branch_any_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_floor_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_ceil_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_sign_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_barrier_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_swizzle_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_i2i_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_i2f_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_f2i_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_f2irnd_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_cmp_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_load_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_store_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_clamp0_max_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_iaddsat_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_imullo0_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_imulhi0_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_idiv0_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_imod_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_txf_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_imadlo0_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_imadlosat0_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_movai_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_iabs_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_leadzero_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_lshift_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_rshift_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_rotate_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_or_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_and_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_xor_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_not_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_bit_extract_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_popcount_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_div_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_atomic_add_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_atomic_xchg_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_atomic_cmp_xchg_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_atomic_min_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_atomic_max_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_atomic_or_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_atomic_and_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_atomic_xor_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_bit_rev_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_texldlpcf_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_texldgpcf_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_conv_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_dp2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_norm_dp2_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_norm_dp3_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_norm_dp4_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_norm_mul_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_img_load_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_img_store_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_tg4_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___extended_instruction_alu_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode___extended_instruction_alu_src0_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_bit_findlsb_gen_0(void *out, struct decode_scope *scope)
{
}
static void decode_bit_findmsb_gen_0(void *out, struct decode_scope *scope)
{
}

void etnaviv_isa_disasm(void *bin, int sz, FILE *out, const struct isa_decode_options *options)
{
    isa_disasm(bin, sz, out, options);
}

bool etnaviv_isa_decode(void *out, void *bin, const struct isa_decode_options *options)
{
    return isa_decode(out, bin, options);
}

uint32_t etnaviv_isa_get_gpu_id(struct decode_scope *scope)
{
    return isa_get_gpu_id(scope);
}
