// MIT License
//
// Copyright (c) 2023-2025 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include "lib/rocprofiler-sdk/kernel_dispatch/kernel_dispatch.hpp"

#include <rocprofiler-sdk/fwd.h>

#include <cstdint>
#include <string_view>
#include <utility>
#include <vector>

namespace rocprofiler
{
namespace kernel_dispatch
{
namespace
{
#define ROCPROFILER_KERNEL_DISPATCH_INFO(CODE)                                                     \
    template <>                                                                                    \
    struct kernel_dispatch_info<ROCPROFILER_##CODE>                                                \
    {                                                                                              \
        static constexpr auto operation_idx = ROCPROFILER_##CODE;                                  \
        static constexpr auto name          = #CODE;                                               \
    };

template <size_t Idx>
struct kernel_dispatch_info;

ROCPROFILER_KERNEL_DISPATCH_INFO(KERNEL_DISPATCH_NONE)
ROCPROFILER_KERNEL_DISPATCH_INFO(KERNEL_DISPATCH_ENQUEUE)
ROCPROFILER_KERNEL_DISPATCH_INFO(KERNEL_DISPATCH_COMPLETE)

template <size_t Idx, size_t... IdxTail>
const char*
name_by_id(const uint32_t id, std::index_sequence<Idx, IdxTail...>)
{
    if(Idx == id) return kernel_dispatch_info<Idx>::name;
    if constexpr(sizeof...(IdxTail) > 0)
        return name_by_id(id, std::index_sequence<IdxTail...>{});
    else
        return nullptr;
}

template <size_t Idx, size_t... IdxTail>
uint32_t
id_by_name(const char* name, std::index_sequence<Idx, IdxTail...>)
{
    if(std::string_view{kernel_dispatch_info<Idx>::name} == std::string_view{name})
        return kernel_dispatch_info<Idx>::operation_idx;
    if constexpr(sizeof...(IdxTail) > 0)
        return id_by_name(name, std::index_sequence<IdxTail...>{});
    else
        return ROCPROFILER_KERNEL_DISPATCH_LAST;
}

template <size_t... Idx>
void
get_ids(std::vector<uint32_t>& _id_list, std::index_sequence<Idx...>)
{
    auto _emplace = [](auto& _vec, uint32_t _v) {
        if(_v < static_cast<uint32_t>(ROCPROFILER_KERNEL_DISPATCH_LAST)) _vec.emplace_back(_v);
    };

    (_emplace(_id_list, kernel_dispatch_info<Idx>::operation_idx), ...);
}

template <size_t... Idx>
void
get_names(std::vector<const char*>& _name_list, std::index_sequence<Idx...>)
{
    auto _emplace = [](auto& _vec, const char* _v) {
        if(_v != nullptr && !std::string_view{_v}.empty()) _vec.emplace_back(_v);
    };

    (_emplace(_name_list, kernel_dispatch_info<Idx>::name), ...);
}
}  // namespace

const char*
name_by_id(uint32_t id)
{
    return name_by_id(id, std::make_index_sequence<ROCPROFILER_KERNEL_DISPATCH_LAST>{});
}

uint32_t
id_by_name(const char* name)
{
    return id_by_name(name, std::make_index_sequence<ROCPROFILER_KERNEL_DISPATCH_LAST>{});
}

std::vector<uint32_t>
get_ids()
{
    auto _data = std::vector<uint32_t>{};
    _data.reserve(ROCPROFILER_KERNEL_DISPATCH_LAST);
    get_ids(_data, std::make_index_sequence<ROCPROFILER_KERNEL_DISPATCH_LAST>{});
    return _data;
}

std::vector<const char*>
get_names()
{
    auto _data = std::vector<const char*>{};
    _data.reserve(ROCPROFILER_KERNEL_DISPATCH_LAST);
    get_names(_data, std::make_index_sequence<ROCPROFILER_KERNEL_DISPATCH_LAST>{});
    return _data;
}
}  // namespace kernel_dispatch
}  // namespace rocprofiler
