// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A5__F226

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_48_0_F_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1150)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xabb1fda3u, 0xba31b72fu, 0, 46 }, // abb1fda3ba31b72f = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1150
    { 0x24c5cc96u, 0x720992b0u, 40, 46 }, // 24c5cc96720992b0 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1150
    { 0xcd653a54u, 0x9247bbf8u, 40, 63 }, // cd653a549247bbf8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1150
    { 0xf962459cu, 0x90e93b36u, 40, 6 }, // f962459c90e93b36 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1150
    { 0x0534be5cu, 0x231c6fdbu, 40, 23 }, // 0534be5c231c6fdb = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1150
    { 0x4e9acda8u, 0x01bb01b8u, 97, 80 }, // 4e9acda801bb01b8 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__32_32__CO__wave2_warp4_stg1--Arch_gfx1150
    { 0x75454cfcu, 0x025a9980u, 97, 155 }, // 75454cfc025a9980 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__32_32__CO__wave3_warp4_stg1--Arch_gfx1150
    { 0xc834a31fu, 0xe0d0877cu, 120, 138 }, // c834a31fe0d0877c = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__64_16__CO__wave1_warp4_stg1--Arch_gfx1150
    { 0x60eacfffu, 0x244d53feu, 120, 80 }, // 60eacfff244d53fe = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__64_16__CO__wave2_warp4_stg1--Arch_gfx1150
    { 0x2201f40bu, 0x6f044d65u, 120, 155 }, // 2201f40b6f044d65 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__64_16__CO__wave3_warp4_stg1--Arch_gfx1150
    { 0xb29a649cu, 0x82e2cb14u, 120, 103 }, // b29a649c82e2cb14 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1150
    { 0x0171bf66u, 0x6b35cfafu, 126, 138 }, // 0171bf666b35cfaf = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__64_32__CO__wave1_warp4_stg1--Arch_gfx1150
    { 0xc0271effu, 0xdb46c5b9u, 126, 80 }, // c0271effdb46c5b9 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__64_32__CO__wave2_warp4_stg1--Arch_gfx1150
    { 0x9224a12au, 0x2dcca30eu, 126, 155 }, // 9224a12a2dcca30e = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__64_32__CO__wave3_warp4_stg1--Arch_gfx1150
    { 0x39932698u, 0x93555ad4u, 126, 103 }, // 3993269893555ad4 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dq/FONLY__＊bf16@16_48_0_F_T_0___gfx11xx__P__64_32__CO__wave4_warp4_stg1--Arch_gfx1150
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1150_mod0
{{ 0, 6, 5, 6, 6, 5, 6, 5, 6, 5},
 { 7, 4, 0, 0, 2, 0, 3, 5, 6, 6},
 { 1, 1, 2, 2, 4, 3, 1,14,13,14},
 { 4, 3, 2, 3, 3, 2, 1, 2, 2,11},
 { 1, 4, 1, 3, 3, 2, 3,14,13,14},
 { 4, 1, 1, 1, 1, 1, 2,12,13,14},
 { 4, 1, 1, 1, 4, 2, 1,14,12,13},
 { 9, 9, 7, 3, 3, 3, 2,13,12,12},
 { 7, 8, 1, 3, 3, 3, 2,13,14,13},
 {10, 4, 1, 2, 1, 1, 1, 1,14,12}}
// End of GPU gfx1150_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A5__F226 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 0;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 0;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 48
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 48 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

