{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- HMACs should be used when producing a cookie or hash based on data
-- and a key. Simple mechanisms for using SHA1 and other algorithms to
-- digest a key and data together are vulnerable to various security
-- issues.
-- <http://en.wikipedia.org/wiki/HMAC HMAC>
-- uses algorithms like SHA1 in a secure way to produce a digest of a
-- key and data.
-- 
-- Both the key and data are arbitrary byte arrays of bytes or characters.
-- 
-- Support for HMAC Digests has been added in GLib 2.30, and support for SHA-512
-- in GLib 2.42. Support for SHA-384 was added in GLib 2.52.
-- 
-- To create a new @GHmac@, use 'GI.GLib.Structs.Hmac.hmacNew'. To free a @GHmac@, use
-- 'GI.GLib.Structs.Hmac.hmacUnref'.
-- 
-- /Since: 2.30/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.Hmac
    ( 

-- * Exported types
    Hmac(..)                                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [copy]("GI.GLib.Structs.Hmac#g:method:copy"), [ref]("GI.GLib.Structs.Hmac#g:method:ref"), [unref]("GI.GLib.Structs.Hmac#g:method:unref"), [update]("GI.GLib.Structs.Hmac#g:method:update").
-- 
-- ==== Getters
-- [getDigest]("GI.GLib.Structs.Hmac#g:method:getDigest"), [getString]("GI.GLib.Structs.Hmac#g:method:getString").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveHmacMethod                       ,
#endif

-- ** copy #method:copy#

#if defined(ENABLE_OVERLOADING)
    HmacCopyMethodInfo                      ,
#endif
    hmacCopy                                ,


-- ** getDigest #method:getDigest#

#if defined(ENABLE_OVERLOADING)
    HmacGetDigestMethodInfo                 ,
#endif
    hmacGetDigest                           ,


-- ** getString #method:getString#

#if defined(ENABLE_OVERLOADING)
    HmacGetStringMethodInfo                 ,
#endif
    hmacGetString                           ,


-- ** new #method:new#

    hmacNew                                 ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    HmacRefMethodInfo                       ,
#endif
    hmacRef                                 ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    HmacUnrefMethodInfo                     ,
#endif
    hmacUnref                               ,


-- ** update #method:update#

#if defined(ENABLE_OVERLOADING)
    HmacUpdateMethodInfo                    ,
#endif
    hmacUpdate                              ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.GLib.Enums as GLib.Enums

#else
import {-# SOURCE #-} qualified GI.GLib.Enums as GLib.Enums

#endif

-- | Memory-managed wrapper type.
newtype Hmac = Hmac (SP.ManagedPtr Hmac)
    deriving (Eq)

instance SP.ManagedPtrNewtype Hmac where
    toManagedPtr (Hmac p) = p

foreign import ccall "g_hmac_get_type" c_g_hmac_get_type :: 
    IO GType

type instance O.ParentTypes Hmac = '[]
instance O.HasParentTypes Hmac

instance B.Types.TypedObject Hmac where
    glibType = c_g_hmac_get_type

instance B.Types.GBoxed Hmac

-- | Convert t'Hmac' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Hmac) where
    gvalueGType_ = c_g_hmac_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr Hmac)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr Hmac)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed Hmac ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Hmac
type instance O.AttributeList Hmac = HmacAttributeList
type HmacAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method Hmac::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "digest_type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "ChecksumType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the desired type of digest"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the key for the HMAC"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key_len"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of the keys"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "key_len"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of the keys"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Hmac" })
-- throws : False
-- Skip return : False

foreign import ccall "g_hmac_new" g_hmac_new :: 
    CUInt ->                                -- digest_type : TInterface (Name {namespace = "GLib", name = "ChecksumType"})
    Ptr Word8 ->                            -- key : TCArray False (-1) 2 (TBasicType TUInt8)
    FCT.CSize ->                            -- key_len : TBasicType TSize
    IO (Ptr Hmac)

-- | Creates a new t'GI.GLib.Structs.Hmac.Hmac', using the digest algorithm /@digestType@/.
-- If the /@digestType@/ is not known, 'P.Nothing' is returned.
-- A t'GI.GLib.Structs.Hmac.Hmac' can be used to compute the HMAC of a key and an
-- arbitrary binary blob, using different hashing algorithms.
-- 
-- A t'GI.GLib.Structs.Hmac.Hmac' works by feeding a binary blob through 'GI.GLib.Structs.Hmac.hmacUpdate'
-- until the data is complete; the digest can then be extracted
-- using 'GI.GLib.Structs.Hmac.hmacGetString', which will return the checksum as a
-- hexadecimal string; or 'GI.GLib.Structs.Hmac.hmacGetDigest', which will return a
-- array of raw bytes. Once either 'GI.GLib.Structs.Hmac.hmacGetString' or
-- 'GI.GLib.Structs.Hmac.hmacGetDigest' have been called on a t'GI.GLib.Structs.Hmac.Hmac', the HMAC
-- will be closed and it won\'t be possible to call 'GI.GLib.Structs.Hmac.hmacUpdate'
-- on it anymore.
-- 
-- Support for digests of type 'GI.GLib.Enums.ChecksumTypeSha512' has been added in GLib 2.42.
-- Support for 'GI.GLib.Enums.ChecksumTypeSha384' was added in GLib 2.52.
-- 
-- /Since: 2.30/
hmacNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GLib.Enums.ChecksumType
    -- ^ /@digestType@/: the desired type of digest
    -> ByteString
    -- ^ /@key@/: the key for the HMAC
    -> m (Maybe Hmac)
    -- ^ __Returns:__ the newly created t'GI.GLib.Structs.Hmac.Hmac', or 'P.Nothing'.
    --   Use 'GI.GLib.Structs.Hmac.hmacUnref' to free the memory allocated by it.
hmacNew digestType key = liftIO $ do
    let keyLen = fromIntegral $ B.length key
    let digestType' = (fromIntegral . fromEnum) digestType
    key' <- packByteString key
    result <- g_hmac_new digestType' key' keyLen
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Hmac) result'
        return result''
    freeMem key'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method Hmac::copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "hmac"
--           , argType = TInterface Name { namespace = "GLib" , name = "Hmac" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GHmac to copy" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Hmac" })
-- throws : False
-- Skip return : False

foreign import ccall "g_hmac_copy" g_hmac_copy :: 
    Ptr Hmac ->                             -- hmac : TInterface (Name {namespace = "GLib", name = "Hmac"})
    IO (Ptr Hmac)

-- | Copies a t'GI.GLib.Structs.Hmac.Hmac'. If /@hmac@/ has been closed, by calling
-- 'GI.GLib.Structs.Hmac.hmacGetString' or 'GI.GLib.Structs.Hmac.hmacGetDigest', the copied
-- HMAC will be closed as well.
-- 
-- /Since: 2.30/
hmacCopy ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Hmac
    -- ^ /@hmac@/: the t'GI.GLib.Structs.Hmac.Hmac' to copy
    -> m Hmac
    -- ^ __Returns:__ the copy of the passed t'GI.GLib.Structs.Hmac.Hmac'. Use 'GI.GLib.Structs.Hmac.hmacUnref'
    --   when finished using it.
hmacCopy hmac = liftIO $ do
    hmac' <- unsafeManagedPtrGetPtr hmac
    result <- g_hmac_copy hmac'
    checkUnexpectedReturnNULL "hmacCopy" result
    result' <- (wrapBoxed Hmac) result
    touchManagedPtr hmac
    return result'

#if defined(ENABLE_OVERLOADING)
data HmacCopyMethodInfo
instance (signature ~ (m Hmac), MonadIO m) => O.OverloadedMethod HmacCopyMethodInfo Hmac signature where
    overloadedMethod = hmacCopy

instance O.OverloadedMethodInfo HmacCopyMethodInfo Hmac where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Hmac.hmacCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Hmac.html#v:hmacCopy"
        })


#endif

-- method Hmac::get_digest
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "hmac"
--           , argType = TInterface Name { namespace = "GLib" , name = "Hmac" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GHmac" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "buffer"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "output buffer" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "digest_len"
--           , argType = TBasicType TSize
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "an inout parameter. The caller initializes it to the\n  size of @buffer. After the call it contains the length of the digest"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "digest_len"
--              , argType = TBasicType TSize
--              , direction = DirectionInout
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "an inout parameter. The caller initializes it to the\n  size of @buffer. After the call it contains the length of the digest"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_hmac_get_digest" g_hmac_get_digest :: 
    Ptr Hmac ->                             -- hmac : TInterface (Name {namespace = "GLib", name = "Hmac"})
    Ptr Word8 ->                            -- buffer : TCArray False (-1) 2 (TBasicType TUInt8)
    Ptr FCT.CSize ->                        -- digest_len : TBasicType TSize
    IO ()

-- | Gets the digest from /@checksum@/ as a raw binary array and places it
-- into /@buffer@/. The size of the digest depends on the type of checksum.
-- 
-- Once this function has been called, the t'GI.GLib.Structs.Hmac.Hmac' is closed and can
-- no longer be updated with 'GI.GLib.Structs.Checksum.checksumUpdate'.
-- 
-- /Since: 2.30/
hmacGetDigest ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Hmac
    -- ^ /@hmac@/: a t'GI.GLib.Structs.Hmac.Hmac'
    -> ByteString
    -- ^ /@buffer@/: output buffer
    -> m ()
hmacGetDigest hmac buffer = liftIO $ do
    let digestLen = fromIntegral $ B.length buffer
    hmac' <- unsafeManagedPtrGetPtr hmac
    buffer' <- packByteString buffer
    digestLen' <- allocMem :: IO (Ptr FCT.CSize)
    poke digestLen' digestLen
    g_hmac_get_digest hmac' buffer' digestLen'
    digestLen'' <- peek digestLen'
    touchManagedPtr hmac
    freeMem buffer'
    freeMem digestLen'
    return ()

#if defined(ENABLE_OVERLOADING)
data HmacGetDigestMethodInfo
instance (signature ~ (ByteString -> m ()), MonadIO m) => O.OverloadedMethod HmacGetDigestMethodInfo Hmac signature where
    overloadedMethod = hmacGetDigest

instance O.OverloadedMethodInfo HmacGetDigestMethodInfo Hmac where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Hmac.hmacGetDigest",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Hmac.html#v:hmacGetDigest"
        })


#endif

-- method Hmac::get_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "hmac"
--           , argType = TInterface Name { namespace = "GLib" , name = "Hmac" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GHmac" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_hmac_get_string" g_hmac_get_string :: 
    Ptr Hmac ->                             -- hmac : TInterface (Name {namespace = "GLib", name = "Hmac"})
    IO CString

-- | Gets the HMAC as a hexadecimal string.
-- 
-- Once this function has been called the t'GI.GLib.Structs.Hmac.Hmac' can no longer be
-- updated with 'GI.GLib.Structs.Hmac.hmacUpdate'.
-- 
-- The hexadecimal characters will be lower case.
-- 
-- /Since: 2.30/
hmacGetString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Hmac
    -- ^ /@hmac@/: a t'GI.GLib.Structs.Hmac.Hmac'
    -> m T.Text
    -- ^ __Returns:__ the hexadecimal representation of the HMAC. The
    --   returned string is owned by the HMAC and should not be modified
    --   or freed.
hmacGetString hmac = liftIO $ do
    hmac' <- unsafeManagedPtrGetPtr hmac
    result <- g_hmac_get_string hmac'
    checkUnexpectedReturnNULL "hmacGetString" result
    result' <- cstringToText result
    touchManagedPtr hmac
    return result'

#if defined(ENABLE_OVERLOADING)
data HmacGetStringMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod HmacGetStringMethodInfo Hmac signature where
    overloadedMethod = hmacGetString

instance O.OverloadedMethodInfo HmacGetStringMethodInfo Hmac where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Hmac.hmacGetString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Hmac.html#v:hmacGetString"
        })


#endif

-- method Hmac::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "hmac"
--           , argType = TInterface Name { namespace = "GLib" , name = "Hmac" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid #GHmac" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Hmac" })
-- throws : False
-- Skip return : False

foreign import ccall "g_hmac_ref" g_hmac_ref :: 
    Ptr Hmac ->                             -- hmac : TInterface (Name {namespace = "GLib", name = "Hmac"})
    IO (Ptr Hmac)

-- | Atomically increments the reference count of /@hmac@/ by one.
-- 
-- This function is MT-safe and may be called from any thread.
-- 
-- /Since: 2.30/
hmacRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Hmac
    -- ^ /@hmac@/: a valid t'GI.GLib.Structs.Hmac.Hmac'
    -> m Hmac
    -- ^ __Returns:__ the passed in t'GI.GLib.Structs.Hmac.Hmac'.
hmacRef hmac = liftIO $ do
    hmac' <- unsafeManagedPtrGetPtr hmac
    result <- g_hmac_ref hmac'
    checkUnexpectedReturnNULL "hmacRef" result
    result' <- (wrapBoxed Hmac) result
    touchManagedPtr hmac
    return result'

#if defined(ENABLE_OVERLOADING)
data HmacRefMethodInfo
instance (signature ~ (m Hmac), MonadIO m) => O.OverloadedMethod HmacRefMethodInfo Hmac signature where
    overloadedMethod = hmacRef

instance O.OverloadedMethodInfo HmacRefMethodInfo Hmac where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Hmac.hmacRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Hmac.html#v:hmacRef"
        })


#endif

-- method Hmac::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "hmac"
--           , argType = TInterface Name { namespace = "GLib" , name = "Hmac" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GHmac" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_hmac_unref" g_hmac_unref :: 
    Ptr Hmac ->                             -- hmac : TInterface (Name {namespace = "GLib", name = "Hmac"})
    IO ()

-- | Atomically decrements the reference count of /@hmac@/ by one.
-- 
-- If the reference count drops to 0, all keys and values will be
-- destroyed, and all memory allocated by the hash table is released.
-- This function is MT-safe and may be called from any thread.
-- Frees the memory allocated for /@hmac@/.
-- 
-- /Since: 2.30/
hmacUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Hmac
    -- ^ /@hmac@/: a t'GI.GLib.Structs.Hmac.Hmac'
    -> m ()
hmacUnref hmac = liftIO $ do
    hmac' <- B.ManagedPtr.disownBoxed hmac
    g_hmac_unref hmac'
    touchManagedPtr hmac
    return ()

#if defined(ENABLE_OVERLOADING)
data HmacUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod HmacUnrefMethodInfo Hmac signature where
    overloadedMethod = hmacUnref

instance O.OverloadedMethodInfo HmacUnrefMethodInfo Hmac where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Hmac.hmacUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Hmac.html#v:hmacUnref"
        })


#endif

-- method Hmac::update
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "hmac"
--           , argType = TInterface Name { namespace = "GLib" , name = "Hmac" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GHmac" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "buffer used to compute the checksum"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "size of the buffer, or -1 if it is a nul-terminated string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "size of the buffer, or -1 if it is a nul-terminated string"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_hmac_update" g_hmac_update :: 
    Ptr Hmac ->                             -- hmac : TInterface (Name {namespace = "GLib", name = "Hmac"})
    Ptr Word8 ->                            -- data : TCArray False (-1) 2 (TBasicType TUInt8)
    DI.Int64 ->                             -- length : TBasicType TSSize
    IO ()

-- | Feeds /@data@/ into an existing t'GI.GLib.Structs.Hmac.Hmac'.
-- 
-- The HMAC must still be open, that is 'GI.GLib.Structs.Hmac.hmacGetString' or
-- 'GI.GLib.Structs.Hmac.hmacGetDigest' must not have been called on /@hmac@/.
-- 
-- /Since: 2.30/
hmacUpdate ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Hmac
    -- ^ /@hmac@/: a t'GI.GLib.Structs.Hmac.Hmac'
    -> ByteString
    -- ^ /@data@/: buffer used to compute the checksum
    -> m ()
hmacUpdate hmac data_ = liftIO $ do
    let length_ = fromIntegral $ B.length data_
    hmac' <- unsafeManagedPtrGetPtr hmac
    data_' <- packByteString data_
    g_hmac_update hmac' data_' length_
    touchManagedPtr hmac
    freeMem data_'
    return ()

#if defined(ENABLE_OVERLOADING)
data HmacUpdateMethodInfo
instance (signature ~ (ByteString -> m ()), MonadIO m) => O.OverloadedMethod HmacUpdateMethodInfo Hmac signature where
    overloadedMethod = hmacUpdate

instance O.OverloadedMethodInfo HmacUpdateMethodInfo Hmac where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.Hmac.hmacUpdate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-Hmac.html#v:hmacUpdate"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveHmacMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveHmacMethod "copy" o = HmacCopyMethodInfo
    ResolveHmacMethod "ref" o = HmacRefMethodInfo
    ResolveHmacMethod "unref" o = HmacUnrefMethodInfo
    ResolveHmacMethod "update" o = HmacUpdateMethodInfo
    ResolveHmacMethod "getDigest" o = HmacGetDigestMethodInfo
    ResolveHmacMethod "getString" o = HmacGetStringMethodInfo
    ResolveHmacMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveHmacMethod t Hmac, O.OverloadedMethod info Hmac p) => OL.IsLabel t (Hmac -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveHmacMethod t Hmac, O.OverloadedMethod info Hmac p, R.HasField t Hmac p) => R.HasField t Hmac p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveHmacMethod t Hmac, O.OverloadedMethodInfo info Hmac) => OL.IsLabel t (O.MethodProxy info Hmac) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


