// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package secret

import (
	"context"
	"strings"

	"github.com/drone/drone/core"
)

// Combine combines the secret services, allowing the system
// to get pipeline secrets from multiple sources.
func Combine(services ...core.SecretService) core.SecretService {
	return &combined{services}
}

type combined struct {
	sources []core.SecretService
}

func (c *combined) Find(ctx context.Context, in *core.SecretArgs) (*core.Secret, error) {
	// Ignore any requests for the .docker/config.json file.
	// This file is reserved for internal use only, and is
	// never exposed to the build environment.
	if isDockerConfig(in.Name) {
		return nil, nil
	}

	for _, source := range c.sources {
		secret, err := source.Find(ctx, in)
		if err != nil {
			return nil, err
		}
		if secret == nil {
			continue
		}
		// if the secret object is not nil, but is empty
		// we should assume the secret service returned a
		// 204 no content, and proceed to the next service
		// in the chain.
		if secret.Data == "" {
			continue
		}
		return secret, nil
	}
	return nil, nil
}

// helper function returns true if the build event matches the
// docker_auth_config variable name.
func isDockerConfig(name string) bool {
	return strings.EqualFold(name, "docker_auth_config") ||
		strings.EqualFold(name, ".dockerconfigjson") ||
		strings.EqualFold(name, ".dockerconfig")
}
