/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QDateTime>
#include <QString>

/*!
 * @brief Net speed counting.
 */
class NetworkSpeed {
public:
	/*!
	 * @brief Constructor.
	 */
	NetworkSpeed(void);

	NetworkSpeed(const NetworkSpeed &other);
#ifdef Q_COMPILER_RVALUE_REFS
	NetworkSpeed(NetworkSpeed &&other) Q_DECL_NOEXCEPT;
#endif /* Q_COMPILER_RVALUE_REFS */

	NetworkSpeed &operator=(const NetworkSpeed &other) Q_DECL_NOTHROW;
#ifdef Q_COMPILER_RVALUE_REFS
	NetworkSpeed &operator=(NetworkSpeed &&other) Q_DECL_NOTHROW;
#endif /* Q_COMPILER_RVALUE_REFS */

	/*!
	 * @brief Compute speeds and generate string.
	 *
	 * @return String containing total of transferred data and speed in
	 *     last interval.
	 */
	QString speedOverview(void) const;

	/*!
	 * @brief Compute speeds and generate string decorated with download
	 *     arrow.
	 *
	 * @return String containing total of transferred data and speed in
	 *     last interval.
	 */
	QString downloadOverview(void) const;

	/*!
	 * @brief Compute speeds and generate string decorated with upload
	 *     arrow.
	 *
	 * @return String containing total of transferred data and speed in
	 *     last interval.
	 */
	QString uploadOverview(void) const;

	/*!
	 * @brief Create a counter.
	 *
	 * @note Stores current time as start time and \a firstTotal
	 *
	 * @param[in] firstTotal First data total. Negative values are
	 *                       treated as 0.
	 * @return Initialised net speed counter.
	 */
	static
	NetworkSpeed createCounter(qint64 firstTotal);

	/*!
	 * @brief Reset a counter.
	 *
	 * @note Equivalent to createCounter().
	 *
	 * @param[in] firstTotal First data total. Negative values are
	 *                       treated as 0.
	 * @return Initialised net speed counter.
	 */
	void resetCounter(qint64 firstTotal);

	/*!
	 * @brief Update counters.
	 *
	 * @note Updates are performed in intervals longer than
	 *     m_updateIntervalMSec.
	 *
	 * @param[in] currentTotal Last total value.
	 */
	void updateCounter(qint64 currentTotal);

private:
	int m_updateIntervalMSec; /*!< Brief update interval in milliseconds. */

	QDateTime m_start; /*!< Time of the first increment. */
	QDateTime m_penultimate; /*!< Time of the last but one (second Last) increment. */
	QDateTime m_stop; /*! Time of the last increment. */
	qint64 m_lastIncrement; /*!< Last increment size (bytes). */
	qint64 m_totalSum; /*< Total increment sum (bytes). */
};
