use std::net::{SocketAddr, TcpListener};

use axum::{Json, Router, debug_handler, extract::State, routing::post};
use clap::Parser;
use color_eyre::eyre::{Context, Result};
use listenfd::ListenFd;
use reqwest::Body;
use tokio::sync::mpsc::UnboundedSender;
use tokio_util::codec::{BytesCodec, FramedRead};

use buildbtw_poc::{PipelineTarget, ScheduleBuild, build_package::build_path};

use crate::args::{Args, Command};

mod args;
mod tasks;

#[derive(Clone)]
struct AppState {
    worker_sender: UnboundedSender<tasks::Message>,
}

#[debug_handler]
async fn schedule_build(
    State(state): State<AppState>,
    Json(body): Json<ScheduleBuild>,
) -> Json<()> {
    state
        .worker_sender
        .send(tasks::Message::BuildPackage(body))
        .wrap_err("Failed to dispatch worker job")
        .unwrap();

    // TODO: return a proper response that can fail?
    Json(())
}

#[tokio::main]
async fn main() -> Result<()> {
    let args = Args::parse();
    // log warnings by default
    buildbtw_poc::tracing::init(args.verbose + 1, args.tokio_console_telemetry);
    color_eyre::install()?;
    tracing::debug!("{args:?}");

    match args.command {
        Command::Run {
            interface,
            port,
            modify_gpg_keyring,
        } => {
            let worker_sender = tasks::start(modify_gpg_keyring);
            let app = Router::new()
                .route("/build/schedule", post(schedule_build))
                .with_state(AppState { worker_sender });

            let mut listenfd = ListenFd::from_env();
            // if listenfd doesn't take a TcpListener (i.e. we're not running via
            // the command above), we fall back to explicitly binding to a given
            // host:port.
            let tcp_listener = if let Some(listener) = listenfd.take_tcp_listener(0).unwrap() {
                listener
            } else {
                let addr = SocketAddr::from((interface, port));
                TcpListener::bind(addr).unwrap()
            };

            axum_server::from_tcp(tcp_listener)
                .serve(app.into_make_service_with_connect_info::<SocketAddr>())
                .await?;
        }
    }
    Ok(())
}

async fn set_build_status(
    status: buildbtw_poc::PackageBuildStatus,
    ScheduleBuild {
        iteration,
        source,
        architecture,
        ..
    }: &ScheduleBuild,
) -> Result<()> {
    let data = buildbtw_poc::SetBuildStatus { status };
    let PipelineTarget { pkgbase, .. } = source;

    reqwest::Client::new()
        .patch(format!(
            "http://0.0.0.0:8080/iteration/{iteration}/pkgbase/{pkgbase}/architecture/{architecture}/status"
        ))
        .json(&data)
        .send()
        .await
        .wrap_err("Failed to send to server")?
        .error_for_status()?;

    tracing::info!("Sent build status to server");

    Ok(())
}

async fn upload_packages(
    ScheduleBuild {
        iteration,
        source,
        architecture,
        package_file_names,
        ..
    }: &ScheduleBuild,
) -> Result<()> {
    for (pkgname, package_file_name) in package_file_names {
        // Build path to the file we'll send
        let dir = build_path(*iteration, &source.pkgbase);
        let path = dir.join(package_file_name);

        // Convert path into async stream body
        let file = tokio::fs::File::open(&path).await.wrap_err(path)?;
        let stream = FramedRead::new(file, BytesCodec::new());
        let body = Body::wrap_stream(stream);

        let PipelineTarget { pkgbase, .. } = source;

        reqwest::Client::new()
        .post(format!(
            "http://0.0.0.0:8080/iteration/{iteration}/pkgbase/{pkgbase}/pkgname/{pkgname}/architecture/{architecture}/package"
        )).body(body).send().await?.error_for_status()?;
    }

    Ok(())
}
