/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.tez.runtime.api.events;

import java.nio.ByteBuffer;

import org.apache.hadoop.classification.InterfaceAudience.Private;
import org.apache.hadoop.classification.InterfaceAudience.Public;
import org.apache.tez.runtime.api.Event;

/**
 * A convenience class to specify multiple DataMovementEvents which share the same payload.
 * This event is generated by Edge in Application Master and sent to downstream vertex input
 * for optimization purpose. It is not to be consumed by any user code.
 */
@Public
public final class CompositeRoutedDataMovementEvent extends Event {

  /**
   * Index(i) of the i-th (physical) Input or Output that generated an Event.
   * For a Processor-generated event, this is ignored.
   */
  private final int sourceIndex;

  /**
   * Index(i) of the i-th (physical) Input or Output that is meant to receive
   * this Event. For a Processor event, this is ignored.
   */
  private int targetIndex;
  private int count;

  /**
   * User Payload for this Event
   */
  private final ByteBuffer userPayload;

  /**
   * Version number to indicate what attempt generated this Event
   */
  private int version;


  @Private
  public static CompositeRoutedDataMovementEvent create(int sourceIndex,
                                                        int targetIndex,
                                                        int count,
                                                        int version,
                                                        ByteBuffer userPayload) {
    return new CompositeRoutedDataMovementEvent(sourceIndex, targetIndex, count, version, userPayload);
  }

  @Private
  CompositeRoutedDataMovementEvent(int sourceIndex,
                                   int targetIndex,
                                   int count,
                                   int version,
                                   ByteBuffer userPayload) {
    this.userPayload = userPayload;
    this.sourceIndex = sourceIndex;
    this.version = version;
    this.targetIndex = targetIndex;
    this.count = count;
  }

  public ByteBuffer getUserPayload() {
    return userPayload == null ? null : userPayload.asReadOnlyBuffer();
  }

  public int getSourceIndex() {
    return sourceIndex;
  }

  public int getTargetIndex() {
    return targetIndex;
  }

  @Private
  public void setTargetIndex(int targetIndex) {
    this.targetIndex = targetIndex;
  }

  public int getCount() {
    return count;
  }

  @Private
  public void setCount(int count) {
    this.count = count;
  }

  public int getVersion() {
    return version;
  }

  @Private
  public void setVersion(int version) {
    this.version = version;
  }

  @Override
  public String toString() {
    return "CompositeRoutedDataMovementEvent [sourceIndex=" + sourceIndex + ", targetIndex="
        + targetIndex + ", count=" + count + ", version=" + version + "]";
  }

  @Private
  public DataMovementEvent expand(int offset) {
    return DataMovementEvent.create(sourceIndex + offset, targetIndex + offset, version, userPayload);
  }
}
