/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.crypto;

import com.google.crypto.tink.subtle.AesGcmJce;
import com.google.crypto.tink.subtle.Hkdf;
import com.google.crypto.tink.subtle.Random;
import com.google.crypto.tink.subtle.X25519;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import java.io.Closeable;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.Arrays;
import java.util.Properties;
import javax.crypto.spec.SecretKeySpec;
import org.apache.spark.network.crypto.AuthMessage;
import org.apache.spark.network.crypto.CtrTransportCipher;
import org.apache.spark.network.crypto.GcmTransportCipher;
import org.apache.spark.network.crypto.TransportCipher;
import org.apache.spark.network.util.TransportConf;
import org.sparkproject.guava.annotations.VisibleForTesting;
import org.sparkproject.guava.base.Preconditions;
import org.sparkproject.guava.primitives.Bytes;

class AuthEngine
implements Closeable {
    public static final byte[] DERIVED_KEY_INFO = "derivedKey".getBytes(StandardCharsets.UTF_8);
    public static final byte[] INPUT_IV_INFO = "inputIv".getBytes(StandardCharsets.UTF_8);
    public static final byte[] OUTPUT_IV_INFO = "outputIv".getBytes(StandardCharsets.UTF_8);
    private static final String MAC_ALGORITHM = "HMACSHA256";
    private static final String LEGACY_CIPHER_ALGORITHM = "AES/CTR/NoPadding";
    private static final String CIPHER_ALGORITHM = "AES/GCM/NoPadding";
    private static final int AES_GCM_KEY_SIZE_BYTES = 16;
    private static final byte[] EMPTY_TRANSCRIPT = new byte[0];
    private static final int UNSAFE_SKIP_HKDF_VERSION = 1;
    private final String appId;
    private final byte[] preSharedSecret;
    private final TransportConf conf;
    private final Properties cryptoConf;
    private final boolean unsafeSkipFinalHkdf;
    private byte[] clientPrivateKey;
    private TransportCipher sessionCipher;

    AuthEngine(String appId, String preSharedSecret, TransportConf conf) {
        Preconditions.checkNotNull(appId);
        Preconditions.checkNotNull(preSharedSecret);
        this.appId = appId;
        this.preSharedSecret = preSharedSecret.getBytes(StandardCharsets.UTF_8);
        this.conf = conf;
        this.cryptoConf = conf.cryptoConf();
        this.unsafeSkipFinalHkdf = conf.authEngineVersion() == 1;
    }

    @VisibleForTesting
    void setClientPrivateKey(byte[] privateKey) {
        this.clientPrivateKey = privateKey;
    }

    private AuthMessage encryptEphemeralPublicKey(byte[] ephemeralX25519PublicKey, byte[] transcript) throws GeneralSecurityException {
        byte[] nonSecretSalt = Random.randBytes((int)16);
        byte[] aadState = Bytes.concat(this.appId.getBytes(StandardCharsets.UTF_8), nonSecretSalt, transcript);
        byte[] derivedKeyEncryptingKey = Hkdf.computeHkdf((String)MAC_ALGORITHM, (byte[])this.preSharedSecret, (byte[])nonSecretSalt, (byte[])aadState, (int)16);
        byte[] aesGcmCiphertext = new AesGcmJce(derivedKeyEncryptingKey).encrypt(ephemeralX25519PublicKey, aadState);
        return new AuthMessage(this.appId, nonSecretSalt, aesGcmCiphertext);
    }

    private byte[] decryptEphemeralPublicKey(AuthMessage encryptedPublicKey, byte[] transcript) throws GeneralSecurityException {
        Preconditions.checkArgument(this.appId.equals(encryptedPublicKey.appId));
        byte[] aadState = Bytes.concat(this.appId.getBytes(StandardCharsets.UTF_8), encryptedPublicKey.salt, transcript);
        byte[] derivedKeyEncryptingKey = Hkdf.computeHkdf((String)MAC_ALGORITHM, (byte[])this.preSharedSecret, (byte[])encryptedPublicKey.salt, (byte[])aadState, (int)16);
        return new AesGcmJce(derivedKeyEncryptingKey).decrypt(encryptedPublicKey.ciphertext, aadState);
    }

    AuthMessage challenge() throws GeneralSecurityException {
        this.setClientPrivateKey(X25519.generatePrivateKey());
        return this.encryptEphemeralPublicKey(X25519.publicFromPrivate((byte[])this.clientPrivateKey), EMPTY_TRANSCRIPT);
    }

    AuthMessage response(AuthMessage encryptedClientPublicKey) throws GeneralSecurityException {
        Preconditions.checkArgument(this.appId.equals(encryptedClientPublicKey.appId));
        byte[] clientPublicKey = this.decryptEphemeralPublicKey(encryptedClientPublicKey, EMPTY_TRANSCRIPT);
        byte[] serverEphemeralPrivateKey = X25519.generatePrivateKey();
        AuthMessage ephemeralServerPublicKey = this.encryptEphemeralPublicKey(X25519.publicFromPrivate((byte[])serverEphemeralPrivateKey), this.getTranscript(encryptedClientPublicKey));
        byte[] sharedSecret = X25519.computeSharedSecret((byte[])serverEphemeralPrivateKey, (byte[])clientPublicKey);
        byte[] challengeResponseTranscript = this.getTranscript(encryptedClientPublicKey, ephemeralServerPublicKey);
        this.sessionCipher = this.generateTransportCipher(sharedSecret, false, challengeResponseTranscript);
        return ephemeralServerPublicKey;
    }

    void deriveSessionCipher(AuthMessage encryptedClientPublicKey, AuthMessage encryptedServerPublicKey) throws GeneralSecurityException {
        Preconditions.checkArgument(this.appId.equals(encryptedClientPublicKey.appId));
        Preconditions.checkArgument(this.appId.equals(encryptedServerPublicKey.appId));
        byte[] serverPublicKey = this.decryptEphemeralPublicKey(encryptedServerPublicKey, this.getTranscript(encryptedClientPublicKey));
        byte[] sharedSecret = X25519.computeSharedSecret((byte[])this.clientPrivateKey, (byte[])serverPublicKey);
        byte[] challengeResponseTranscript = this.getTranscript(encryptedClientPublicKey, encryptedServerPublicKey);
        this.sessionCipher = this.generateTransportCipher(sharedSecret, true, challengeResponseTranscript);
    }

    private TransportCipher generateTransportCipher(byte[] sharedSecret, boolean isClient, byte[] transcript) throws GeneralSecurityException {
        byte[] derivedKey = this.unsafeSkipFinalHkdf ? sharedSecret : Hkdf.computeHkdf((String)MAC_ALGORITHM, (byte[])sharedSecret, (byte[])transcript, (byte[])DERIVED_KEY_INFO, (int)16);
        byte[] clientIv = Hkdf.computeHkdf((String)MAC_ALGORITHM, (byte[])sharedSecret, (byte[])transcript, (byte[])INPUT_IV_INFO, (int)16);
        byte[] serverIv = Hkdf.computeHkdf((String)MAC_ALGORITHM, (byte[])sharedSecret, (byte[])transcript, (byte[])OUTPUT_IV_INFO, (int)16);
        SecretKeySpec sessionKey = new SecretKeySpec(derivedKey, "AES");
        if (LEGACY_CIPHER_ALGORITHM.equalsIgnoreCase(this.conf.cipherTransformation())) {
            return new CtrTransportCipher(this.cryptoConf, sessionKey, isClient ? clientIv : serverIv, isClient ? serverIv : clientIv);
        }
        if (CIPHER_ALGORITHM.equalsIgnoreCase(this.conf.cipherTransformation())) {
            return new GcmTransportCipher(sessionKey);
        }
        throw new IllegalArgumentException(String.format("Unsupported cipher mode: %s. %s and %s are supported.", this.conf.cipherTransformation(), CIPHER_ALGORITHM, LEGACY_CIPHER_ALGORITHM));
    }

    private byte[] getTranscript(AuthMessage ... encryptedPublicKeys) {
        ByteBuf transcript = Unpooled.buffer((int)Arrays.stream(encryptedPublicKeys).mapToInt(k -> k.encodedLength()).sum());
        Arrays.stream(encryptedPublicKeys).forEachOrdered(k -> k.encode(transcript));
        return transcript.array();
    }

    TransportCipher sessionCipher() {
        Preconditions.checkState(this.sessionCipher != null);
        return this.sessionCipher;
    }

    @Override
    public void close() {
    }
}

