/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.kafka.clients.consumer.CommitFailedException;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.errors.TaskCorruptedException;
import org.apache.kafka.streams.errors.TaskMigratedException;
import org.apache.kafka.streams.internals.StreamsConfigUtils;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.StreamTask;
import org.apache.kafka.streams.processor.internals.Task;
import org.apache.kafka.streams.processor.internals.TaskExecutionMetadata;
import org.apache.kafka.streams.processor.internals.TaskManager;
import org.apache.kafka.streams.processor.internals.TasksRegistry;
import org.slf4j.Logger;

public class TaskExecutor {
    private final Logger log;
    private final TasksRegistry tasks;
    private final TaskManager taskManager;
    private final TaskExecutionMetadata executionMetadata;

    public TaskExecutor(TasksRegistry tasks, TaskManager taskManager, TaskExecutionMetadata executionMetadata, LogContext logContext) {
        this.tasks = tasks;
        this.taskManager = taskManager;
        this.executionMetadata = executionMetadata;
        this.log = logContext.logger(this.getClass());
    }

    int process(int maxNumRecords, Time time) {
        int totalProcessed = 0;
        Task lastProcessed = null;
        for (Task task : this.tasks.activeTasks()) {
            long now = time.milliseconds();
            try {
                if (!this.executionMetadata.canProcessTask(task, now)) continue;
                lastProcessed = task;
                totalProcessed += this.processTask(task, maxNumRecords, now, time);
            }
            catch (Throwable t) {
                this.executionMetadata.registerTaskError(task, t, now);
                this.executionMetadata.removeTaskFromSuccessfullyProcessedBeforeClosing(lastProcessed);
                this.commitSuccessfullyProcessedTasks();
                throw t;
            }
        }
        return totalProcessed;
    }

    private int processTask(Task task, int maxNumRecords, long begin, Time time) {
        int processed;
        long now;
        long then = now = begin;
        try {
            for (processed = 0; processed < maxNumRecords && task.process(now); ++processed) {
                task.clearTaskTimeout();
            }
            if (processed > 0 && this.executionMetadata.hasNamedTopologies() && this.executionMetadata.processingMode() != StreamsConfigUtils.ProcessingMode.EXACTLY_ONCE_V2) {
                this.log.trace("Successfully processed task {}", (Object)task.id());
                this.executionMetadata.addToSuccessfullyProcessed(task);
            }
        }
        catch (TimeoutException timeoutException) {
            task.maybeInitTaskTimeoutOrThrow(now, (Exception)((Object)timeoutException));
            this.log.error(String.format("Could not complete processing records for %s due to the following exception; will move to next task and retry later", task.id()), (Throwable)timeoutException);
        }
        catch (TaskMigratedException e) {
            this.log.info("Failed to process stream task {} since it got migrated to another thread already. Will trigger a new rebalance and close all tasks as zombies together.", (Object)task.id());
            throw e;
        }
        catch (StreamsException e) {
            this.log.error(String.format("Failed to process stream task %s due to the following error:", task.id()), (Throwable)((Object)e));
            e.setTaskId(task.id());
            throw e;
        }
        catch (RuntimeException e) {
            this.log.error(String.format("Failed to process stream task %s due to the following error:", task.id()), (Throwable)e);
            throw new StreamsException(e, task.id());
        }
        finally {
            now = time.milliseconds();
            task.recordProcessBatchTime(now - then);
        }
        return processed;
    }

    int commitTasksAndMaybeUpdateCommittableOffsets(Collection<Task> tasksToCommit, Map<Task, Map<TopicPartition, OffsetAndMetadata>> consumedOffsetsAndMetadata) {
        int committed = 0;
        for (Task task : tasksToCommit) {
            Map<TopicPartition, OffsetAndMetadata> offsetAndMetadata;
            if (!task.commitNeeded() || (offsetAndMetadata = task.prepareCommit()).isEmpty()) continue;
            consumedOffsetsAndMetadata.put(task, offsetAndMetadata);
        }
        this.commitOffsetsOrTransaction(consumedOffsetsAndMetadata);
        for (Task task : tasksToCommit) {
            if (!task.commitNeeded()) continue;
            task.clearTaskTimeout();
            ++committed;
            task.postCommit(false);
        }
        return committed;
    }

    void commitOffsetsOrTransaction(Map<Task, Map<TopicPartition, OffsetAndMetadata>> offsetsPerTask) {
        this.log.debug("Committing task offsets {}", offsetsPerTask.entrySet().stream().collect(Collectors.toMap(t -> ((Task)t.getKey()).id(), Map.Entry::getValue)));
        HashSet<TaskId> corruptedTasks = new HashSet<TaskId>();
        if (this.executionMetadata.processingMode() == StreamsConfigUtils.ProcessingMode.EXACTLY_ONCE_ALPHA) {
            for (Task task2 : this.taskManager.activeRunningTaskIterable()) {
                Map<TopicPartition, OffsetAndMetadata> taskOffsetsToCommit = offsetsPerTask.getOrDefault(task2, Collections.emptyMap());
                if (taskOffsetsToCommit.isEmpty() && !this.taskManager.streamsProducerForTask(task2.id()).transactionInFlight()) continue;
                try {
                    this.taskManager.streamsProducerForTask(task2.id()).commitTransaction(taskOffsetsToCommit, this.taskManager.consumerGroupMetadata());
                    this.updateTaskCommitMetadata(taskOffsetsToCommit);
                }
                catch (TimeoutException timeoutException) {
                    this.log.error(String.format("Committing task %s failed.", task2.id()), (Throwable)timeoutException);
                    corruptedTasks.add(task2.id());
                }
            }
        } else if (this.executionMetadata.processingMode() == StreamsConfigUtils.ProcessingMode.EXACTLY_ONCE_V2) {
            if (!offsetsPerTask.isEmpty() || this.taskManager.threadProducer().transactionInFlight()) {
                Map<TopicPartition, OffsetAndMetadata> allOffsets = offsetsPerTask.values().stream().flatMap(e -> e.entrySet().stream()).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
                try {
                    this.taskManager.threadProducer().commitTransaction(allOffsets, this.taskManager.consumerGroupMetadata());
                    this.updateTaskCommitMetadata(allOffsets);
                }
                catch (TimeoutException timeoutException) {
                    this.log.error(String.format("Committing task(s) %s failed.", offsetsPerTask.keySet().stream().map(t -> t.id().toString()).collect(Collectors.joining(", "))), (Throwable)timeoutException);
                    offsetsPerTask.keySet().forEach(task -> corruptedTasks.add(task.id()));
                }
            }
        } else if (!offsetsPerTask.isEmpty()) {
            Map<TopicPartition, OffsetAndMetadata> allOffsets = offsetsPerTask.values().stream().flatMap(e -> e.entrySet().stream()).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            try {
                this.taskManager.consumerCommitSync(allOffsets);
                this.updateTaskCommitMetadata(allOffsets);
            }
            catch (CommitFailedException error) {
                throw new TaskMigratedException("Consumer committing offsets failed, indicating the corresponding thread is no longer part of the group", error);
            }
            catch (TimeoutException timeoutException) {
                this.log.error(String.format("Committing task(s) %s failed.", offsetsPerTask.keySet().stream().map(t -> t.id().toString()).collect(Collectors.joining(", "))), (Throwable)timeoutException);
                throw timeoutException;
            }
            catch (KafkaException error) {
                throw new StreamsException("Error encountered committing offsets via consumer", error);
            }
        }
        if (!corruptedTasks.isEmpty()) {
            throw new TaskCorruptedException(corruptedTasks);
        }
    }

    private void updateTaskCommitMetadata(Map<TopicPartition, OffsetAndMetadata> allOffsets) {
        if (!allOffsets.isEmpty()) {
            for (Task task : this.tasks.activeTasks()) {
                if (!(task instanceof StreamTask)) continue;
                for (TopicPartition topicPartition : task.inputPartitions()) {
                    if (!allOffsets.containsKey(topicPartition)) continue;
                    ((StreamTask)task).updateCommittedOffsets(topicPartition, allOffsets.get(topicPartition).offset());
                }
            }
        }
    }

    private void commitSuccessfullyProcessedTasks() {
        if (!this.executionMetadata.successfullyProcessed().isEmpty()) {
            this.log.info("Streams encountered an error when processing tasks. Will commit all previously successfully processed tasks {}", this.executionMetadata.successfullyProcessed().stream().map(Task::id));
            this.commitTasksAndMaybeUpdateCommittableOffsets(this.executionMetadata.successfullyProcessed(), new HashMap<Task, Map<TopicPartition, OffsetAndMetadata>>());
        }
        this.executionMetadata.clearSuccessfullyProcessed();
    }

    int punctuate() {
        int punctuated = 0;
        for (Task task : this.tasks.activeTasks()) {
            try {
                if (!this.executionMetadata.canPunctuateTask(task)) continue;
                if (task.maybePunctuateStreamTime()) {
                    ++punctuated;
                }
                if (!task.maybePunctuateSystemTime()) continue;
                ++punctuated;
            }
            catch (TaskMigratedException e) {
                this.log.info("Failed to punctuate stream task {} since it got migrated to another thread already. Will trigger a new rebalance and close all tasks as zombies together.", (Object)task.id());
                throw e;
            }
            catch (StreamsException e) {
                this.log.error("Failed to punctuate stream task {} due to the following error:", (Object)task.id(), (Object)e);
                e.setTaskId(task.id());
                throw e;
            }
            catch (KafkaException e) {
                this.log.error("Failed to punctuate stream task {} due to the following error:", (Object)task.id(), (Object)e);
                throw new StreamsException(e, task.id());
            }
        }
        return punctuated;
    }
}

