/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.mongo.wrapper.cursor;

import com.mongodb.ServerAddress;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCursor;
import org.apache.hop.mongo.MongoDbException;
import org.bson.Document;

public class DefaultCursorWrapper implements MongoCursorWrapper {
  private final FindIterable<Document> findIterable;
  private MongoCursor<Document> cursor;
  private ServerAddress serverAddress;

  public DefaultCursorWrapper(FindIterable<Document> findIterable) {
    this.findIterable = findIterable;
    this.cursor = null;
  }

  private MongoCursor<Document> getCursor() {
    if (cursor == null) {
      cursor = findIterable.iterator();
    }
    return cursor;
  }

  @Override
  public boolean hasNext() throws MongoDbException {
    return getCursor().hasNext();
  }

  @Override
  public Document next() throws MongoDbException {
    Document doc = getCursor().next();
    // Capture server address from first result
    if (serverAddress == null) {
      serverAddress = getCursor().getServerAddress();
    }
    return doc;
  }

  @Override
  public ServerAddress getServerAddress() throws MongoDbException {
    if (serverAddress != null) {
      return serverAddress;
    }
    return getCursor().getServerAddress();
  }

  @Override
  public void close() throws MongoDbException {
    if (cursor != null) {
      cursor.close();
    }
  }

  @Override
  public MongoCursorWrapper limit(int i) throws MongoDbException {
    // Close existing cursor if open
    if (cursor != null) {
      cursor.close();
      cursor = null;
    }
    return new DefaultCursorWrapper(findIterable.limit(i));
  }
}
