/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hertzbeat.warehouse.store.history.tsdb.duckdb;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.temporal.TemporalAmount;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.hertzbeat.common.entity.arrow.RowWrapper;
import org.apache.hertzbeat.common.entity.dto.Value;
import org.apache.hertzbeat.common.entity.message.CollectRep;
import org.apache.hertzbeat.common.util.JsonUtil;
import org.apache.hertzbeat.common.util.TimePeriodUtil;
import org.apache.hertzbeat.warehouse.store.history.tsdb.AbstractHistoryDataStorage;
import org.apache.hertzbeat.warehouse.store.history.tsdb.duckdb.DuckdbProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

@Component
@ConditionalOnProperty(prefix="warehouse.store.duckdb", name={"enabled"}, havingValue="true")
public class DuckdbDatabaseDataStorage
extends AbstractHistoryDataStorage {
    private static final Logger log = LoggerFactory.getLogger(DuckdbDatabaseDataStorage.class);
    private static final String DRIVER_NAME = "org.duckdb.DuckDBDriver";
    private static final String URL_PREFIX = "jdbc:duckdb:";
    private static final int TARGET_CHART_POINTS = 800;
    private static final Pattern DAY_PATTERN = Pattern.compile("^(\\d+)[dD]$");
    private final String expireTimeStr;
    private final String dbPath;
    private HikariDataSource dataSource;

    public DuckdbDatabaseDataStorage(DuckdbProperties duckdbProperties) {
        this.dbPath = duckdbProperties.storePath();
        this.expireTimeStr = duckdbProperties.expireTime();
        this.serverAvailable = this.initDuckDb();
        if (this.serverAvailable) {
            this.startExpiredDataCleaner();
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private boolean initDuckDb() {
        try {
            Class.forName(DRIVER_NAME);
            HikariConfig config = new HikariConfig();
            config.setJdbcUrl(URL_PREFIX + this.dbPath);
            config.setDriverClassName(DRIVER_NAME);
            config.setPoolName("DuckDB-Pool");
            config.setMinimumIdle(1);
            config.setMaximumPoolSize(10);
            config.setConnectionTimeout(30000L);
            config.setConnectionTestQuery("SELECT 1");
            this.dataSource = new HikariDataSource(config);
            try (Connection connection = this.dataSource.getConnection();){
                boolean bl;
                block15: {
                    Statement statement = connection.createStatement();
                    try {
                        String createTableSql = "CREATE TABLE IF NOT EXISTS hzb_history (\ninstance VARCHAR,\napp VARCHAR,\nmetrics VARCHAR,\nmetric VARCHAR,\nmetric_type SMALLINT,\nint32_value INTEGER,\ndouble_value DOUBLE,\nstr_value VARCHAR,\nrecord_time BIGINT,\nlabels VARCHAR)";
                        statement.execute(createTableSql);
                        statement.execute("CREATE INDEX IF NOT EXISTS idx_hzb_history_composite ON hzb_history (instance, app, metrics, metric, record_time)");
                        statement.execute("CREATE INDEX IF NOT EXISTS idx_hzb_history_record_time ON hzb_history (record_time)");
                        bl = true;
                        if (statement == null) break block15;
                    }
                    catch (Throwable throwable) {
                        if (statement != null) {
                            try {
                                statement.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    statement.close();
                }
                return bl;
            }
        }
        catch (Exception e) {
            log.error("Failed to init duckdb: {}", (Object)e.getMessage(), (Object)e);
            if (this.dataSource != null) {
                this.dataSource.close();
            }
            return false;
        }
    }

    private void startExpiredDataCleaner() {
        ScheduledExecutorService scheduledExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread thread = new Thread(r, "duckdb-cleaner");
            thread.setDaemon(true);
            return thread;
        });
        scheduledExecutor.scheduleAtFixedRate(() -> {
            long expireTime;
            log.info("[duckdb] start data cleaner and checkpoint...");
            try {
                String cleanExpireStr = this.expireTimeStr == null ? "" : this.expireTimeStr.trim();
                Matcher dayMatcher = DAY_PATTERN.matcher(cleanExpireStr);
                if (NumberUtils.isParsable((String)cleanExpireStr)) {
                    expireTime = NumberUtils.toLong((String)cleanExpireStr);
                    expireTime = (ZonedDateTime.now().toEpochSecond() - expireTime) * 1000L;
                } else if (dayMatcher.matches()) {
                    long days = Long.parseLong(dayMatcher.group(1));
                    ZonedDateTime dateTime = ZonedDateTime.now().minus(Duration.ofDays(days));
                    expireTime = dateTime.toEpochSecond() * 1000L;
                } else {
                    TemporalAmount temporalAmount = TimePeriodUtil.parseTokenTime((String)cleanExpireStr);
                    ZonedDateTime dateTime = ZonedDateTime.now().minus(temporalAmount);
                    expireTime = dateTime.toEpochSecond() * 1000L;
                }
            }
            catch (Exception e) {
                log.error("expiredDataCleaner time error: {}. use default expire time to clean: 30d", (Object)e.getMessage());
                ZonedDateTime dateTime = ZonedDateTime.now().minus(Duration.ofDays(30L));
                expireTime = dateTime.toEpochSecond() * 1000L;
            }
            try (Connection connection = this.dataSource.getConnection();){
                try (Statement statement = connection.prepareStatement("DELETE FROM hzb_history WHERE record_time < ?");){
                    statement.setLong(1, expireTime);
                    int rows = statement.executeUpdate();
                    if (rows > 0) {
                        log.info("[duckdb] delete {} expired records.", (Object)rows);
                    }
                }
                statement = connection.createStatement();
                try {
                    statement.execute("CHECKPOINT");
                }
                finally {
                    if (statement != null) {
                        statement.close();
                    }
                }
            }
            catch (Exception e) {
                log.error("[duckdb] clean expired data error: {}", (Object)e.getMessage(), (Object)e);
            }
        }, 5L, 60L, TimeUnit.MINUTES);
    }

    @Override
    public void saveData(CollectRep.MetricsData metricsData) {
        if (!this.isServerAvailable() || metricsData.getCode() != CollectRep.Code.SUCCESS || metricsData.getValues().isEmpty()) {
            return;
        }
        String monitorType = metricsData.getApp();
        String metrics = metricsData.getMetrics();
        String insertSql = "INSERT INTO hzb_history VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        try (Connection connection = this.dataSource.getConnection();
             PreparedStatement preparedStatement = connection.prepareStatement(insertSql);){
            RowWrapper rowWrapper = metricsData.readRow();
            HashMap labels = new HashMap();
            while (rowWrapper.hasNextRow()) {
                rowWrapper = rowWrapper.nextRow();
                long time = metricsData.getTime();
                rowWrapper.cellStream().forEach(cell -> {
                    if (cell.getMetadataAsBoolean("label").booleanValue()) {
                        labels.put(cell.getField().getName(), cell.getValue());
                    }
                });
                String labelsJson = JsonUtil.toJson(labels);
                rowWrapper.cellStream().forEach(cell -> {
                    try {
                        String metric = cell.getField().getName();
                        String columnValue = cell.getValue();
                        int fieldType = cell.getMetadataAsInteger("type");
                        preparedStatement.setString(1, metricsData.getInstance());
                        preparedStatement.setString(2, monitorType);
                        preparedStatement.setString(3, metrics);
                        preparedStatement.setString(4, metric);
                        if ("&nbsp;".equals(columnValue)) {
                            preparedStatement.setShort(5, (short)0);
                            preparedStatement.setObject(6, null);
                            preparedStatement.setObject(7, null);
                            preparedStatement.setObject(8, null);
                        } else {
                            switch (fieldType) {
                                case 1: {
                                    preparedStatement.setShort(5, (short)1);
                                    preparedStatement.setObject(6, null);
                                    preparedStatement.setObject(7, null);
                                    preparedStatement.setString(8, columnValue);
                                    break;
                                }
                                case 3: {
                                    preparedStatement.setShort(5, (short)3);
                                    preparedStatement.setInt(6, Integer.parseInt(columnValue));
                                    preparedStatement.setObject(7, null);
                                    preparedStatement.setObject(8, null);
                                    break;
                                }
                                default: {
                                    preparedStatement.setShort(5, (short)0);
                                    preparedStatement.setObject(6, null);
                                    double v = Double.parseDouble(columnValue);
                                    preparedStatement.setDouble(7, v);
                                    preparedStatement.setObject(8, null);
                                }
                            }
                        }
                        preparedStatement.setLong(9, time);
                        preparedStatement.setString(10, labelsJson);
                        preparedStatement.addBatch();
                    }
                    catch (SQLException e) {
                        log.error("error setting prepared statement", (Throwable)e);
                    }
                });
                labels.clear();
            }
            preparedStatement.executeBatch();
        }
        catch (Exception e) {
            log.error("[duckdb] save data error: {}", (Object)e.getMessage(), (Object)e);
        }
    }

    @Override
    public Map<String, List<Value>> getHistoryMetricData(String instance, String app, String metrics, String metric, String history) {
        HashMap<String, List<Value>> instanceValuesMap = new HashMap<String, List<Value>>(8);
        if (!this.isServerAvailable()) {
            return instanceValuesMap;
        }
        StringBuilder sqlBuilder = new StringBuilder("SELECT record_time,\nmetric_type,\nint32_value,\ndouble_value,\nstr_value,\nlabels FROM hzb_history\nWHERE instance = ?\nAND app = ?\nAND metrics = ?\nAND metric = ?\n");
        long timeBefore = 0L;
        if (history != null) {
            try {
                TemporalAmount temporalAmount = TimePeriodUtil.parseTokenTime((String)history);
                ZonedDateTime dateTime = ZonedDateTime.now().minus(temporalAmount);
                timeBefore = dateTime.toEpochSecond() * 1000L;
                sqlBuilder.append(" AND record_time >= ?");
            }
            catch (Exception e) {
                log.error("parse history time error: {}", (Object)e.getMessage());
            }
        }
        sqlBuilder.append(" ORDER BY record_time DESC LIMIT 20000");
        try (Connection connection = this.dataSource.getConnection();
             PreparedStatement preparedStatement = connection.prepareStatement(sqlBuilder.toString());){
            preparedStatement.setString(1, instance);
            preparedStatement.setString(2, app);
            preparedStatement.setString(3, metrics);
            preparedStatement.setString(4, metric);
            if (timeBefore > 0L) {
                preparedStatement.setLong(5, timeBefore);
            }
            try (ResultSet resultSet = preparedStatement.executeQuery();){
                while (resultSet.next()) {
                    long time = resultSet.getLong("record_time");
                    short type = resultSet.getShort("metric_type");
                    String labels = resultSet.getString("labels");
                    String value = this.formatValue(type, resultSet);
                    List valueList = instanceValuesMap.computeIfAbsent(labels == null ? "" : labels, k -> new LinkedList());
                    valueList.add(new Value(value, time));
                }
            }
        }
        catch (SQLException e) {
            log.error("[duckdb] query data error: {}", (Object)e.getMessage(), (Object)e);
        }
        return instanceValuesMap;
    }

    @Override
    public Map<String, List<Value>> getHistoryIntervalMetricData(String instance, String app, String metrics, String metric, String history) {
        HashMap<String, List<Value>> instanceValuesMap = new HashMap<String, List<Value>>(8);
        if (!this.isServerAvailable()) {
            return instanceValuesMap;
        }
        long timeBefore = 0L;
        long endTime = System.currentTimeMillis();
        long interval = 0L;
        if (history != null) {
            try {
                TemporalAmount temporalAmount = TimePeriodUtil.parseTokenTime((String)history);
                ZonedDateTime dateTime = ZonedDateTime.now().minus(temporalAmount);
                timeBefore = dateTime.toEpochSecond() * 1000L;
                interval = (endTime - timeBefore) / 800L;
                if (interval < 60000L) {
                    interval = 60000L;
                }
            }
            catch (Exception e) {
                log.error("parse history time error: {}", (Object)e.getMessage());
                timeBefore = System.currentTimeMillis() - 86400000L;
                interval = 60000L;
            }
        }
        String sql = "SELECT\nCAST(record_time / ? AS BIGINT) * ? AS ts_bucket,\nAVG(double_value) AS avg_val,\nMIN(double_value) AS min_val,\nMAX(double_value) AS max_val,\nFIRST(str_value) AS str_val,\nmetric_type, labels\nFROM hzb_history\nWHERE instance = ? AND app = ? AND metrics = ? AND metric = ? AND record_time >= ?\nGROUP BY ts_bucket, metric_type, labels\nORDER BY ts_bucket";
        try (Connection connection = this.dataSource.getConnection();
             PreparedStatement preparedStatement = connection.prepareStatement(sql);){
            preparedStatement.setLong(1, interval);
            preparedStatement.setLong(2, interval);
            preparedStatement.setString(3, instance);
            preparedStatement.setString(4, app);
            preparedStatement.setString(5, metrics);
            preparedStatement.setString(6, metric);
            preparedStatement.setLong(7, timeBefore);
            try (ResultSet resultSet = preparedStatement.executeQuery();){
                while (resultSet.next()) {
                    Value valueObj;
                    long time = resultSet.getLong("ts_bucket");
                    short type = resultSet.getShort("metric_type");
                    String labels = resultSet.getString("labels");
                    if (type == 0) {
                        double avg = resultSet.getDouble("avg_val");
                        double min = resultSet.getDouble("min_val");
                        double max = resultSet.getDouble("max_val");
                        if (resultSet.wasNull()) {
                            valueObj = new Value(null, time);
                        } else {
                            String avgStr = BigDecimal.valueOf(avg).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
                            String minStr = BigDecimal.valueOf(min).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
                            String maxStr = BigDecimal.valueOf(max).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
                            valueObj = new Value(avgStr, time);
                            valueObj.setMean(avgStr);
                            valueObj.setMin(minStr);
                            valueObj.setMax(maxStr);
                        }
                    } else {
                        String strVal = resultSet.getString("str_val");
                        valueObj = new Value(strVal, time);
                    }
                    List valueList = instanceValuesMap.computeIfAbsent(labels == null ? "" : labels, k -> new LinkedList());
                    valueList.add(valueObj);
                }
            }
        }
        catch (SQLException e) {
            log.error("[duckdb] query interval data error: {}", (Object)e.getMessage(), (Object)e);
        }
        return instanceValuesMap;
    }

    private String formatValue(int type, ResultSet resultSet) throws SQLException {
        if (type == 0) {
            double v = resultSet.getDouble("double_value");
            if (!resultSet.wasNull()) {
                return BigDecimal.valueOf(v).setScale(4, RoundingMode.HALF_UP).stripTrailingZeros().toPlainString();
            }
        } else if (type == 3) {
            int v = resultSet.getInt("int32_value");
            if (!resultSet.wasNull()) {
                return String.valueOf(v);
            }
        } else {
            return resultSet.getString("str_value");
        }
        return "";
    }

    public void destroy() throws Exception {
        if (this.dataSource != null) {
            this.dataSource.close();
        }
    }
}

