/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.tserver.compactions;

import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;
import org.apache.accumulo.core.client.TableNotFoundException;
import org.apache.accumulo.core.client.admin.compaction.CompactableFile;
import org.apache.accumulo.core.conf.ConfigurationTypeHelper;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.data.NamespaceId;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.data.TabletId;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.dataImpl.TabletIdImpl;
import org.apache.accumulo.core.spi.common.ServiceEnvironment;
import org.apache.accumulo.core.spi.compaction.CompactionExecutorId;
import org.apache.accumulo.core.spi.compaction.CompactionJob;
import org.apache.accumulo.core.spi.compaction.CompactionKind;
import org.apache.accumulo.core.spi.compaction.CompactionPlan;
import org.apache.accumulo.core.spi.compaction.CompactionPlanner;
import org.apache.accumulo.core.spi.compaction.CompactionServiceId;
import org.apache.accumulo.core.util.compaction.CompactionExecutorIdImpl;
import org.apache.accumulo.core.util.compaction.CompactionPlanImpl;
import org.apache.accumulo.core.util.compaction.CompactionPlannerInitParams;
import org.apache.accumulo.core.util.ratelimit.RateLimiter;
import org.apache.accumulo.core.util.ratelimit.SharedRateLimiterFactory;
import org.apache.accumulo.core.util.threads.ThreadPoolNames;
import org.apache.accumulo.core.util.threads.ThreadPools;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.server.ServiceEnvironmentImpl;
import org.apache.accumulo.tserver.compactions.Compactable;
import org.apache.accumulo.tserver.compactions.CompactionExecutor;
import org.apache.accumulo.tserver.compactions.ExternalCompactionExecutor;
import org.apache.accumulo.tserver.compactions.InternalCompactionExecutor;
import org.apache.accumulo.tserver.compactions.ProvisionalCompactionPlanner;
import org.apache.accumulo.tserver.compactions.SubmittedJob;
import org.apache.accumulo.tserver.metrics.CompactionExecutorsMetrics;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionService {
    private CompactionPlanner planner;
    private Map<CompactionExecutorId, CompactionExecutor> executors;
    private final CompactionServiceId myId;
    private final Map<KeyExtent, Collection<SubmittedJob>> submittedJobs = new ConcurrentHashMap<KeyExtent, Collection<SubmittedJob>>();
    private final ServerContext context;
    private String plannerClassName;
    private Map<String, String> plannerOpts;
    private final CompactionExecutorsMetrics ceMetrics;
    private final ExecutorService planningExecutor;
    private final Map<CompactionKind, ConcurrentMap<KeyExtent, Compactable>> queuedForPlanning;
    private final RateLimiter readLimiter;
    private final RateLimiter writeLimiter;
    private final AtomicLong rateLimit = new AtomicLong(0L);
    private final Function<CompactionExecutorId, ExternalCompactionExecutor> externExecutorSupplier;
    private final Cache<TableId, Long> maxScanFilesExceededErrorCache;
    private static final Logger log = LoggerFactory.getLogger(CompactionService.class);

    public CompactionService(String serviceName, String plannerClass, Long maxRate, Map<String, String> plannerOptions, ServerContext context, CompactionExecutorsMetrics ceMetrics, Function<CompactionExecutorId, ExternalCompactionExecutor> externExecutorSupplier) {
        Preconditions.checkArgument((maxRate >= 0L ? 1 : 0) != 0);
        this.myId = CompactionServiceId.of((String)serviceName);
        this.context = context;
        this.plannerClassName = plannerClass;
        this.plannerOpts = plannerOptions;
        this.ceMetrics = ceMetrics;
        this.externExecutorSupplier = externExecutorSupplier;
        CompactionPlannerInitParams initParams = new CompactionPlannerInitParams(this.myId, this.plannerOpts, (ServiceEnvironment)new ServiceEnvironmentImpl(context));
        this.planner = CompactionService.createPlanner(this.myId, plannerClass, plannerOptions, initParams);
        HashMap tmpExecutors = new HashMap();
        this.rateLimit.set(maxRate);
        this.readLimiter = SharedRateLimiterFactory.getInstance((ScheduledThreadPoolExecutor)this.context.getScheduledExecutor()).create("CS_" + serviceName + "_read", () -> this.rateLimit.get());
        this.writeLimiter = SharedRateLimiterFactory.getInstance((ScheduledThreadPoolExecutor)this.context.getScheduledExecutor()).create("CS_" + serviceName + "_write", () -> this.rateLimit.get());
        initParams.getRequestedExecutors().forEach((ceid, numThreads) -> tmpExecutors.put(ceid, new InternalCompactionExecutor((CompactionExecutorId)ceid, (int)numThreads, ceMetrics, this.readLimiter, this.writeLimiter)));
        initParams.getRequestedExternalExecutors().forEach(ceid -> tmpExecutors.put(ceid, (CompactionExecutor)externExecutorSupplier.apply((CompactionExecutorId)ceid)));
        this.executors = Map.copyOf(tmpExecutors);
        this.planningExecutor = ThreadPools.getServerThreadPools().getPoolBuilder(ThreadPoolNames.COMPACTION_SERVICE_COMPACTION_PLANNER_POOL).numCoreThreads(1).numMaxThreads(1).withTimeOut(0L, TimeUnit.MILLISECONDS).build();
        this.queuedForPlanning = new EnumMap<CompactionKind, ConcurrentMap<KeyExtent, Compactable>>(CompactionKind.class);
        for (CompactionKind kind : CompactionKind.values()) {
            this.queuedForPlanning.put(kind, new ConcurrentHashMap());
        }
        this.maxScanFilesExceededErrorCache = CacheBuilder.newBuilder().expireAfterWrite(5L, TimeUnit.MINUTES).build();
        log.debug("Created new compaction service id:{} rate limit:{} planner:{} planner options:{}", new Object[]{this.myId, maxRate, plannerClass, plannerOptions});
    }

    private static CompactionPlanner createPlanner(CompactionServiceId myId, String plannerClass, Map<String, String> options, CompactionPlannerInitParams initParams) {
        try {
            CompactionPlanner planner = (CompactionPlanner)ConfigurationTypeHelper.getClassInstance(null, (String)plannerClass, CompactionPlanner.class);
            planner.init((CompactionPlanner.InitParameters)initParams);
            return planner;
        }
        catch (Exception e) {
            log.error("Failed to create compaction planner for {} using class:{} options:{}.  Compaction service will not start any new compactions until its configuration is fixed.", new Object[]{myId, plannerClass, options, e});
            return new ProvisionalCompactionPlanner(myId);
        }
    }

    private boolean reconcile(Set<CompactionJob> jobs, Collection<SubmittedJob> submitted) {
        for (SubmittedJob submittedJob : submitted) {
            SubmittedJob.Status status = submittedJob.getStatus();
            if (status == SubmittedJob.Status.QUEUED) {
                if (jobs.remove(submittedJob.getJob()) || submittedJob.cancel(SubmittedJob.Status.QUEUED)) continue;
                return false;
            }
            if (status != SubmittedJob.Status.RUNNING) continue;
            for (CompactionJob job : jobs) {
                if (Collections.disjoint(submittedJob.getJob().getFiles(), job.getFiles())) continue;
                return false;
            }
        }
        return true;
    }

    public void submitCompaction(CompactionKind kind, Compactable compactable, Consumer<Compactable> completionCallback) {
        Objects.requireNonNull(compactable);
        if (this.queuedForPlanning.get(kind).putIfAbsent(compactable.getExtent(), compactable) == null) {
            try {
                this.planningExecutor.execute(() -> {
                    try {
                        Optional<Compactable.Files> files = compactable.getFiles(this.myId, kind);
                        if (files.isEmpty() || files.orElseThrow().candidates.isEmpty()) {
                            log.trace("Compactable returned no files {} {} {}", new Object[]{this.myId, compactable.getExtent(), kind});
                        } else {
                            CompactionPlan plan = this.getCompactionPlan(kind, files.orElseThrow(), compactable);
                            this.submitCompactionJob(plan, files.orElseThrow(), compactable, completionCallback);
                        }
                    }
                    finally {
                        this.queuedForPlanning.get(kind).remove(compactable.getExtent());
                    }
                });
            }
            catch (RejectedExecutionException e) {
                this.queuedForPlanning.get(kind).remove(compactable.getExtent());
                throw e;
            }
        }
    }

    private CompactionPlan getCompactionPlan(CompactionKind kind, Compactable.Files files, Compactable compactable) {
        CompactionPlan plan;
        CpPlanParams params = new CpPlanParams(kind, compactable, files);
        log.trace("Planning compactions {} {} {} {} {}", new Object[]{this.myId, this.planner.getClass().getName(), compactable.getExtent(), kind, files});
        try {
            plan = this.planner.makePlan((CompactionPlanner.PlanningParameters)params);
            TableId tableId = compactable.getTableId();
            if (plan.getJobs().isEmpty()) {
                Long last;
                int maxScanFiles = this.context.getTableConfiguration(tableId).getCount(Property.TSERV_SCAN_MAX_OPENFILES);
                if (files.allFiles.size() >= maxScanFiles && files.compacting.isEmpty() && (last = (Long)this.maxScanFilesExceededErrorCache.getIfPresent((Object)tableId)) == null) {
                    log.warn("The tablet {} has {} files and the max files for scan is {}.  No compactions are running and none were planned for this tablet by {}, so the files will not be reduced by compaction which could cause scans to fail.  Please check your compaction configuration. This log message is temporarily suppressed for the entire table.", new Object[]{compactable.getExtent(), files.allFiles.size(), maxScanFiles, this.myId});
                    this.maxScanFilesExceededErrorCache.put((Object)tableId, (Object)System.currentTimeMillis());
                }
            }
        }
        catch (RuntimeException e) {
            log.debug("Planner failed {} {} {} {} {}", new Object[]{this.myId, this.planner.getClass().getName(), compactable.getExtent(), kind, files, e});
            throw e;
        }
        return this.convertPlan(plan, kind, files.allFiles, files.candidates);
    }

    private void submitCompactionJob(CompactionPlan plan, Compactable.Files files, Compactable compactable, Consumer<Compactable> completionCallback) {
        Stream<CompactionExecutorIdImpl> execIds = plan.getJobs().stream().map(cj -> (CompactionExecutorIdImpl)cj.getExecutor());
        if (compactable.getExtent().isMeta() && execIds.anyMatch(ceid -> ceid.isExternalId())) {
            log.error("Compacting metadata tablets on external compactors is not supported, please change config for compaction service ({}) and/or table ASAP.  {} is not compacting, ignoring plan {}", new Object[]{this.myId, compactable.getExtent(), plan});
            return;
        }
        HashSet<CompactionJob> jobs = new HashSet<CompactionJob>(plan.getJobs());
        Collection submitted = this.submittedJobs.getOrDefault(compactable.getExtent(), List.of());
        if (!submitted.isEmpty()) {
            submitted.removeIf(sj -> {
                SubmittedJob.Status status = sj.getStatus();
                return status != SubmittedJob.Status.QUEUED && status != SubmittedJob.Status.RUNNING;
            });
        }
        if (this.reconcile(jobs, submitted)) {
            for (CompactionJob job : jobs) {
                CompactionExecutor executor = this.executors.get(job.getExecutor());
                SubmittedJob submittedJob = executor.submit(this.myId, job, compactable, completionCallback);
                this.submittedJobs.computeIfAbsent(compactable.getExtent(), k -> new ConcurrentLinkedQueue()).add(submittedJob);
            }
            if (!jobs.isEmpty()) {
                log.trace("Submitted compaction plan {} id:{} files:{} plan:{}", new Object[]{compactable.getExtent(), this.myId, files, plan});
            }
        } else {
            log.trace("Did not submit compaction plan {} id:{} files:{} plan:{}", new Object[]{compactable.getExtent(), this.myId, files, plan});
        }
    }

    private CompactionPlan convertPlan(CompactionPlan plan, CompactionKind kind, Set<CompactableFile> allFiles, Set<CompactableFile> candidates) {
        if (plan.getClass().equals(CompactionPlanImpl.class)) {
            return plan;
        }
        CompactionPlanImpl.BuilderImpl builder = new CompactionPlanImpl.BuilderImpl(kind, allFiles, candidates);
        for (CompactionJob job : plan.getJobs()) {
            Preconditions.checkArgument((job.getKind() == kind ? 1 : 0) != 0, (String)"Unexpected compaction kind %s != %s", (Object)job.getKind(), (Object)kind);
            builder.addJob(job.getPriority(), job.getExecutor(), (Collection)job.getFiles());
        }
        return builder.build();
    }

    public boolean isCompactionQueued(KeyExtent extent) {
        return ((Collection)this.submittedJobs.getOrDefault(extent, List.of())).stream().anyMatch(job -> job.getStatus() == SubmittedJob.Status.QUEUED);
    }

    public void configurationChanged(String plannerClassName, Long maxRate, Map<String, String> plannerOptions) {
        Preconditions.checkArgument((maxRate >= 0L ? 1 : 0) != 0);
        long old = this.rateLimit.getAndSet(maxRate);
        if (old != maxRate) {
            log.debug("Updated compaction service id:{} rate limit:{}", (Object)this.myId, (Object)maxRate);
        }
        if (this.plannerClassName.equals(plannerClassName) && this.plannerOpts.equals(plannerOptions)) {
            return;
        }
        CompactionPlannerInitParams initParams = new CompactionPlannerInitParams(this.myId, plannerOptions, (ServiceEnvironment)new ServiceEnvironmentImpl(this.context));
        CompactionPlanner tmpPlanner = CompactionService.createPlanner(this.myId, plannerClassName, plannerOptions, initParams);
        HashMap tmpExecutors = new HashMap();
        initParams.getRequestedExecutors().forEach((ceid, numThreads) -> {
            InternalCompactionExecutor executor = (InternalCompactionExecutor)this.executors.get(ceid);
            if (executor == null) {
                executor = new InternalCompactionExecutor((CompactionExecutorId)ceid, (int)numThreads, this.ceMetrics, this.readLimiter, this.writeLimiter);
            } else {
                executor.setThreads((int)numThreads);
            }
            tmpExecutors.put(ceid, executor);
        });
        initParams.getRequestedExternalExecutors().forEach(ceid -> {
            ExternalCompactionExecutor executor = (ExternalCompactionExecutor)this.executors.get(ceid);
            if (executor == null) {
                executor = this.externExecutorSupplier.apply((CompactionExecutorId)ceid);
            }
            tmpExecutors.put(ceid, executor);
        });
        Sets.difference(this.executors.keySet(), tmpExecutors.keySet()).forEach(ceid -> this.executors.get(ceid).stop());
        this.plannerClassName = plannerClassName;
        this.plannerOpts = plannerOptions;
        this.executors = Map.copyOf(tmpExecutors);
        this.planner = tmpPlanner;
        log.debug("Updated compaction service id:{} planner:{} options:{}", new Object[]{this.myId, plannerClassName, plannerOptions});
    }

    public void stop() {
        this.executors.values().forEach(CompactionExecutor::stop);
        log.debug("Stopped compaction service {}", (Object)this.myId);
    }

    int getCompactionsRunning(CompactionExecutor.CType ctype) {
        return this.executors.values().stream().mapToInt(ce -> ce.getCompactionsRunning(ctype)).sum();
    }

    int getCompactionsQueued(CompactionExecutor.CType ctype) {
        return this.executors.values().stream().mapToInt(ce -> ce.getCompactionsQueued(ctype)).sum();
    }

    public void getExternalExecutorsInUse(Consumer<CompactionExecutorId> idConsumer) {
        this.executors.forEach((ceid, ce) -> {
            if (ce instanceof ExternalCompactionExecutor) {
                idConsumer.accept((CompactionExecutorId)ceid);
            }
        });
    }

    public void compactableClosed(KeyExtent extent) {
        this.executors.values().forEach(compExecutor -> compExecutor.compactableClosed(extent));
    }

    private class CpPlanParams
    implements CompactionPlanner.PlanningParameters {
        private final CompactionKind kind;
        private final Compactable comp;
        private final Compactable.Files files;
        private final ServiceEnvironment senv;

        public CpPlanParams(CompactionKind kind, Compactable comp, Compactable.Files files) {
            this.senv = new ServiceEnvironmentImpl(CompactionService.this.context);
            this.kind = kind;
            this.comp = comp;
            this.files = files;
        }

        public NamespaceId getNamespaceId() throws TableNotFoundException {
            return CompactionService.this.context.getNamespaceId(this.comp.getTableId());
        }

        public TableId getTableId() {
            return this.comp.getTableId();
        }

        public TabletId getTabletId() {
            return new TabletIdImpl(this.comp.getExtent());
        }

        public ServiceEnvironment getServiceEnvironment() {
            return this.senv;
        }

        public double getRatio() {
            return this.comp.getCompactionRatio();
        }

        public CompactionKind getKind() {
            return this.kind;
        }

        public Collection<CompactionJob> getRunningCompactions() {
            return this.files.compacting;
        }

        public Collection<CompactableFile> getCandidates() {
            return this.files.candidates;
        }

        public Collection<CompactableFile> getAll() {
            return this.files.allFiles;
        }

        public Map<String, String> getExecutionHints() {
            if (this.kind == CompactionKind.USER) {
                return this.files.executionHints;
            }
            return Map.of();
        }

        public CompactionPlan.Builder createPlanBuilder() {
            return new CompactionPlanImpl.BuilderImpl(this.kind, this.files.allFiles, this.files.candidates);
        }
    }
}

